/*---------------------------------------------------------
    This file is part of the program suite TEN
    (Tools for Elastic Networks)

    Copyright (C)

	Lars Ackermann
    G. Matthias Ullmann
    Bayreuth 2014

    www.bisb.uni-bayreuth.de

    This program is free software: you can redistribute
    it and/or modify it under the terms of the
    GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will
    be useful, but WITHOUT ANY WARRANTY; without even the
    implied warranty of MERCHANTABILITY or FITNESS FOR A
    PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the
    GNU Affero General Public License along with this
    program.  If not, see <http://www.gnu.org/licenses/>.
-----------------------------------------------------------*/


#include "main.h"
#include <iostream>
#include "FileHandler.h"
#include "SetupConstants.h"
#include "PseudoInvCalculator.h"
#include <stdlib.h>
#include <string.h>
#include <vector>
#include <time.h>

using namespace std;

/*possible arguments in argv:
 * [0] - contains the command itself, which means the location of the executed program.
 * [1] - absolute path (including file name) to the setup file to use (there it is specified which proteins shall be processed [names ...], where they are [paths...]...)
 *
 *settings:
 *	protNames: names of proteins
 *	pqrmPaths: paths for files of the proteins for all protNames (only folder path; NO FILE NAME AT THE END)
 *
 * */
int main(int argc, char **argv) {

	clock_t init, final, allPrevious;
	init = clock();

	char* setupFilePath;
	if (argc < 2) {
		setupFilePath = "settings";
	} else {
		setupFilePath = argv[1];
	}

	SetupConstants* constants = new SetupConstants();
	FileHandler* fileHandler = new FileHandler();
	fileHandler->readSettings(constants, setupFilePath);
	SettingsMap settings = constants->getSettings();
	MassMap allMasses = constants -> getMasses();
	BBMap allBBNumbers = constants -> getBBNumbers();

	vector<string> names = constants -> getNames();
	vector<string> protPaths = constants -> getPaths();

	vector<string> models = constants -> getModels();
	for (uint i = 0; i < (names.size()); i++) {
		allPrevious = clock() - init;
		const char* currentName = names.at(i).c_str();
		std::cout << i+1 << ".: Processing protein " << currentName << ":" << std::endl;
		string currentPath;
		if (i < protPaths.size()) {
			currentPath = protPaths.at(i);
			int pathLength = strlen(currentPath.c_str());
			char ending = currentPath.at(pathLength-1);
			if (ending != '/') {
				currentPath.append("/");
			}
		} else {
			currentPath = "";
		}
		if(settings["bvalAlphaOnly"] == 1){
			fileHandler -> readPQRM(currentPath.c_str(), currentName, allBBNumbers);
		}
		constants -> setAllBBNumbers(allBBNumbers);

		for (uint j = 0; j < models.size(); j++) {
			const char* currentModel = models.at(j).c_str();
			std::cout << "Reading eigenvalues and -vectors...";
			vector< vector<double> > evvcs = fileHandler -> readEigenvaluesAndVecs(currentPath.c_str(),currentName,currentModel, settings, allMasses);
			std::cout << "done" << std::endl;
			constants -> updateSettingsMap(settings);
			constants -> setAllMasses(allMasses);
			AbstractPseudoInvCalculator* pseudoInvCalc = new PseudoInvCalculator(constants,fileHandler);
			vector<double> first = evvcs.at(0);
			double* eigenvals = new double[evvcs.size()];
			//the dimension has to be (first.size()-1)^2 because the first element is the eigenvalue and must be ignored for calculating the size of the eigenvector array.
			double* eigenvecs = new double[(first.size()-1) * (first.size()-1)];
			for(uint k=0; k<evvcs.size(); k++){
				vector<double> evvc = evvcs.at(k);
				//the first element of evvc is the eigenvalue
				eigenvals[k] = evvc.at(0);
				for(uint l=1; l<first.size(); l++){
					eigenvecs[k*(first.size()-1)+l-1] = evvc.at(l);
				}
			}
			std::cout << "Calculate pseudo inverse, covariances, correlations and theoretical b values..." << std::endl;
			pseudoInvCalc -> analyzeData(eigenvals,eigenvecs,currentName,currentPath.c_str(),currentModel,first.size()-1);
			std::cout << "done" << std::endl;
			final = clock() - init - allPrevious;
			std::cout << "Time: " << (double) final / ((double) CLOCKS_PER_SEC) << std::endl;
			delete[] eigenvals;
			delete[] eigenvecs;
			delete pseudoInvCalc;
		}
	}
	delete fileHandler;
	delete constants;
	std::cout << "All done... ";
	final = clock() - init;
	std::cout << "Total time: " << (double) final / ((double) CLOCKS_PER_SEC) << std::endl;
	return 0;
}
