/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @date
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The PseudoInvCalculator class is a implementation of the abstract class \link<AbstractPseudoInvCalculator>AbstractPseudoInvCalculator\endlink which is able to calculate the pseudo inverse, covariances, correlations and b values.
 * It uses Lapack++ for some matrix operations. This in turn uses the original Fortran implementation of
 * Lapack and so Lapack++ is only a wrapper.
 *
 */

#ifndef PSEUDOINVCALCULATOR_H_
#define PSEUDOINVCALCULATOR_H_

#include "SetupConstants.h"
#include "AbstractPseudoInvCalculator.h"

/*
 *
 */
class PseudoInvCalculator: public AbstractPseudoInvCalculator {
public:
	/*!
	 * This is the constructor you have to use if you need an instance of AbstractPseudoInvCalculator. It uses the
	 * super constructor of AbstractPseudoInvCalculator. For more information please have a look at the documentation
	 * of the super constructor.
	 * @param settings
	 * @param fileHandler
	 * @return
	 */
	PseudoInvCalculator(SetupConstants* settings, FileHandler* fileHandler);

	/*!
	 * This is the standard destructor.
	 * @return nothing to return
	 */
	virtual ~PseudoInvCalculator();

	/*!
	 * This method processes the whole analysis of the protein. Step by step the following happens:
	 * <ol>
	 * 	<li>the pseudo inverse of whether the kirchhoff and/or the hessian matrix is done using the method calculatePseudoInverse.</li>
	 * 	<li>the covariances are calculated using the method calculateCovariances, </li>
	 * 	<li>the correlations are calculated using the method calculateCorrelations,</li>
	 * 	<li>the theoretical b values are calculated using the method calculateTheoreticalBValues,</li>
	 * 	<li>the results for step 2,3 and 4 are printed out using the i/o manager.</li>
	 * </ol>
	 * @param eigenvals These are the eigenvalues calculated from the kirchhoff matrix (GNM) or the hessian matrix (ANM).
	 * @param eigenvecs These are the eigenvectors calculated from the kirchhoff matrix (GNM) or the hessian matrix (ANM).
	 * @param protName This is the name of the protein the analysis is done for.
	 * @param model This is the type of the network model the eigenvalues and eigenvectors belong to.
	 * @param matrixDimensions This is the number of atoms (GNM) or 3 x the number of atoms (ANM).
	 */
	virtual void analyzeData(double* eigenvals, double* eigenvecs, const char* protName, const char* path, const char* model, int matrixDimensions);

	/*!
	 * The hessian matrix may be multiplied by M^-1/2 (2 times) which means that the hessian elements are weighted with the atom masses.
	 * In order to calculate the pseudo inverse you have to recalculate the modified hessian matrix. For more information please have a look
	 * at the user manual.
	 * @param hessianMatrix This is the mass weighted hessian matrix.
	 * @param lengthOfEVecs This is the number of atoms (GNM) or 3 x the number of atoms (ANM).
	 * @return The hessian matrix with no orthogonal eigenvectors but instead of that correct mass weighted values. For more information please
	 * have a look at the user manual.
	 */
	virtual double* prepareHessianMatrix(double* hessianMatrix, int lengthOfEVecs, const char* protName);

	/*!
	 * This method calculates the pseudo inverse of a matrix of eigenvectors. To make it easier the BLAS method Blas_Mat_Mat_Mult is used.
	 * It works as follows: <br />
	 * Blas_Mat_Mat_Mult multiplies(const LaGenMatDouble& A, const LaGenMatDouble& B, LaGenMatDouble& C,
	 * bool transpose_A, bool transpose_B = false,	double alpha = 1.0, double beta = 0.0)
	 * multiplies two matrices (first two arguments). The third matrix (3rd argument) is the result matrix.
	 * You are able to decide whether the first and/or the second matrix is to be used transposed (arguments four and five).
	 * The last two arguments are factors. The first is a factor for multiplication with the product of the first two matrices and
	 * the second is the factor for the result matrix. The sense is visible if you keep in mind that this method calculates according\
	 * to the formula: C := alpha*A*B + beta*C
	 * @param eigenvals These are the eigenvalues calculated from the kirchhoff matrix (GNM) or the hessian matrix (ANM).
	 * @param eigenvecs These are the eigenvectors calculated from the kirchhoff matrix (GNM) or the hessian matrix (ANM).
	 * @param model This is the type of the network model the eigenvalues and eigenvectors belong to.
	 * @param lengthOfEVecs This is the number of atoms (GNM) or 3 x the number of atoms (ANM).
	 * @return the pseudo inverse of the given matrix.
	 */
	virtual double* calculatePseudoInverse(double* eigenvals, double* eigenvecs, const char* model, int lengthOfEVecs);

	/*!
	 * This method calculates the covariances of an inverse matrix. To make it easier the method uses the BLAS function Blas_Scale which
	 * works as follows:
	 * A := s * A where s is the scaling factor which is the only thing you need for calculating the covariances.
	 * @param invMatrix This is the pseudo inverse whether of the kirchhoff matrix or the hessian matrix.
	 * @param lengthOfEVecs This is the number of atoms (GNM) or 3 x the number of atoms (ANM).
	 * @param model
	 * @return The covariance matrix for the pseudo inverse.
	 */
	virtual double* calculateCovariances(double* invMatrix, int lengthOfEVecs, const char* model);

	/*!
	 * This method calculates the correlations with the help of a covariance matrix. This method does not use any Lapack++ or BLAS routine.
	 * @param covarianceMatrix This is the covariance matrix of the pseudo inverse.
	 * @param lengthOfEVecs This is the number of atoms the protein has.
	 * @return The correlation matrix.
	 */
	virtual double* calculateCorrelations(double* covarianceMatrix, int lengthOfEVecs);
	/*!
	 * The method calculates theroetical b values for a given covariance matrix. This method does not use any Lapack++ or BLAS routine.
	 * @param covarianceMatrix This is the covariance matrix of the pseudo inverse.
	 * @param lengthOfEVecs This is the number of atoms the protein has.
	 * @return The theroretical b values.
	 */
	virtual vector<double> calculateTheoreticalBValues(double* covarianceMatrix, int lengthOfEVecs, const char* protName);
};

#endif /* PSEUDOINVCALCULATOR_H_ */
