/*---------------------------------------------------------
    This file is part of the program suite TEN
    (Tools for Elastic Networks)

    Copyright (C)

	Lars Ackermann
    G. Matthias Ullmann
    Bayreuth 2014

    www.bisb.uni-bayreuth.de

    This program is free software: you can redistribute
    it and/or modify it under the terms of the
    GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will
    be useful, but WITHOUT ANY WARRANTY; without even the
    implied warranty of MERCHANTABILITY or FITNESS FOR A
    PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the
    GNU Affero General Public License along with this
    program.  If not, see <http://www.gnu.org/licenses/>.
-----------------------------------------------------------*/


#include "PseudoInvCalculator.h"
#include "blas3pp.h"
#include "gmd.h"
#include "math.h"
#include <stdio.h>

PseudoInvCalculator::PseudoInvCalculator(SetupConstants* settings, FileHandler* fileHandler) :
	AbstractPseudoInvCalculator(settings, fileHandler) {
}

PseudoInvCalculator::~PseudoInvCalculator() {
}

void PseudoInvCalculator::analyzeData(double* eigenvals, double* eigenvecs, const char* protName, const char* path, const char* model, int matrixDimensions) {
	if (strncmp(model, "ANM", 3) == 0) {
		matrixDimensions = matrixDimensions / 3;
		SettingsMap setMap = settings -> getSettings();
		if (setMap["massWeightedHessian"] == 1) {
			eigenvecs = this -> prepareHessianMatrix(eigenvecs, matrixDimensions, protName);
		}
	}
	double* pseudoInv = this -> calculatePseudoInverse(eigenvals, eigenvecs, model, matrixDimensions);
	double* covariances = this -> calculateCovariances(pseudoInv, matrixDimensions, model);

	double* correlations = this -> calculateCorrelations(covariances, matrixDimensions);
	vector<double> bfactors = this -> calculateTheoreticalBValues(covariances, matrixDimensions, protName);

	fileHandler -> printCovariances(covariances, protName, path, model, matrixDimensions);
	fileHandler -> printCorrelations(correlations, protName, path, model, matrixDimensions);
	vector<int> bbNumbers = (settings -> getBBNumbers())[protName];
	fileHandler -> printBValues(bfactors, bbNumbers, protName, path, model);
	delete[] covariances;
	delete[] correlations;
	//GNM works works directly with the pseudoInv from line 29. So covariances shares the memory with pseudoInv. Deleting it here would cause a "double free or corruption".
	if (strncmp(model, "GNM", 3) != 0) {
		delete[] pseudoInv;
	}
}

double* PseudoInvCalculator::prepareHessianMatrix(double* modifiedHessianMatrix, int lengthOfEVecs, const char* protName) {
	MassMap allMasses = settings -> getMasses();
	vector<double> masses = allMasses[protName];

	LaGenMatDouble modHessMatr = LaGenMatDouble(modifiedHessianMatrix, lengthOfEVecs * 3, lengthOfEVecs * 3 - 6, false);
	//this step must be done because the hessian matrix has
	for (int i = 0; i < lengthOfEVecs; i++) {
		for (int j = 0; j < lengthOfEVecs * 3 - 6; j++) {
			for (int k = 0; k < 3; k++) {
				modHessMatr(i * 3 + k, j) = modHessMatr(i * 3 + k, j) / sqrt(masses.at(i));
			}
		}
	}
	return modifiedHessianMatrix;
}

double* PseudoInvCalculator::calculatePseudoInverse(double* eigenvals, double* eigenvecs, const char* model, int lengthOfEVecs) {
	int numberOfEValues = 0.0;
	if (strncmp(model, "GNM", 3) == 0) {
		numberOfEValues = lengthOfEVecs - 1;
	} else if (strncmp(model, "ANM", 3) == 0) {
		lengthOfEVecs = lengthOfEVecs * 3;
		numberOfEValues = lengthOfEVecs - 6;
	}
	double* pseudoInv = new double[lengthOfEVecs * lengthOfEVecs];
	for (int i = 0; i < lengthOfEVecs * lengthOfEVecs; i++) {
		pseudoInv[i] = 0;
	}
	LaGenMatDouble eigVecMatr = LaGenMatDouble(eigenvecs, lengthOfEVecs, numberOfEValues, false);
	LaGenMatDouble pseudoInvMatr = LaGenMatDouble(pseudoInv, lengthOfEVecs, lengthOfEVecs, false);

	for (int i = 0; i < numberOfEValues; i++) {
		LaGenMatDouble eigVec = eigVecMatr.col(i);
		double eigValNormalizer = 1 / eigenvals[i];
		/*Blas_Mat_Mat_Mult multiplies(const LaGenMatDouble& A, const LaGenMatDouble& B, LaGenMatDouble& C,	bool transpose_A, bool transpose_B = false,	double alpha = 1.0, double beta = 0.0)
		 multiplies two matrices (first two arguments). The third matrix (3rd argument) is the result matrix.
		 You are able to decide whether the first and/or the second matrix is to be used transposed (arguments four and five).
		 The last two arguments are factors. The first is a factor for multiplication with the product of the first two matrices and
		 the second is the factor for the result matrix. The sense is visible if you keep in mind that this method calculates according
		 to the formula: C := alpha*A*B + beta*C */
		Blas_Mat_Mat_Mult(eigVec, eigVec, pseudoInvMatr, false, true, eigValNormalizer, 1.0);
	}
	return pseudoInv;
}

double* PseudoInvCalculator::calculateCovariances(double* invMatrix, int lengthOfEVecs, const char* model) {
	SettingsMap setMap = settings -> getSettings();
	double temp = setMap["temp"];
	double gasConst = setMap["uniGasConst"];
	if (strncmp(model, "GNM", 3) == 0) {
		LaGenMatDouble laInvMatrix = LaGenMatDouble(invMatrix, lengthOfEVecs, lengthOfEVecs, false);
		Blas_Scale(3 * temp * gasConst, laInvMatrix);
		return invMatrix;
	} else if (strncmp(model, "ANM", 3) == 0) {
		LaGenMatDouble laInvMatrix = LaGenMatDouble(invMatrix, lengthOfEVecs * 3, lengthOfEVecs * 3 - 6, false);
		double* covariances = new double[lengthOfEVecs * lengthOfEVecs];
		LaGenMatDouble covMatrix = LaGenMatDouble(covariances, lengthOfEVecs, lengthOfEVecs, false);
		for (int i = 0; i < lengthOfEVecs; i++) {
			for (int j = 0; j < lengthOfEVecs; j++) {
				//for ANM the trace is used for calculating the
				covMatrix(i, j) = laInvMatrix(i * 3, j * 3) + laInvMatrix(i * 3 + 1, j * 3 + 1) + laInvMatrix(i * 3 + 2, j * 3 + 2);
			}
		}
		//normally it would look like this: Blas_Scale(3*temp*gasConst,covMatrix);
		//The factor 3 must be left out because you accumulate 3 values (the trace) and so you have to
		//calculate the average.
		Blas_Scale(temp * gasConst, covMatrix);
		return covariances;
	}
	perror("The model you have chosen is not supported. Please choose whether GNM, ANM or both.");
	exit(1);
	return NULL;
}

double* PseudoInvCalculator::calculateCorrelations(double* covarianceMatrix, int lengthOfEVecs) {
	double* correlations = new double[lengthOfEVecs * lengthOfEVecs];
	LaGenMatDouble covMatr = LaGenMatDouble(covarianceMatrix, lengthOfEVecs, lengthOfEVecs, false);
	LaGenMatDouble correlMatr = LaGenMatDouble(correlations, lengthOfEVecs, lengthOfEVecs, false);
	for (int i = 0; i < lengthOfEVecs; i++) {
		for (int j = 0; j < lengthOfEVecs; j++) {
			correlMatr(i, j) = covMatr(i, j) / sqrt(covMatr(i, i) * covMatr(j, j));
		}
	}
	return correlations;
}

vector<double> PseudoInvCalculator::calculateTheoreticalBValues(double* covarianceMatrix, int lengthOfEVecs, const char* protName) {
	SettingsMap setMap = settings -> getSettings();
	LaGenMatDouble covMatr = LaGenMatDouble(covarianceMatrix, lengthOfEVecs, lengthOfEVecs, false);
	vector<double> bfactors;
	double bfactorSum = 0;
	if (setMap["bvalAlphaOnly"] == 1) {
		vector<int> bbNumbers = (settings -> getBBNumbers())[protName];
		for (uint i = 0; i < bbNumbers.size(); i++) {
			double curB = 8 * pow(M_PI, 2) / 3 * covMatr(bbNumbers.at(i), bbNumbers.at(i));
			bfactors.push_back(curB);
			bfactorSum += curB;
		}
	} else {
		for (int i = 0; i < lengthOfEVecs; i++) {
			double curB = 8 * pow(M_PI, 2) / 3 * covMatr(i, i);
			bfactors.push_back(curB);
			bfactorSum += curB;
		}
	}
	double averageB = bfactorSum/bfactors.size();
	std::cout << "Average b value: " << averageB << std::endl;
	return bfactors;
}
