/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @date
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The FileHandler class is responsible for input and output actions. It reads the
 * user's settings file, the files with the eigenvalues and -vectors. As output actions there are
 * printing out covariances, correlations and the b values.
 */

#ifndef FILEHANDLER_H
#define FILEHANDLER_H
#include "SetupConstants.h"

class FileHandler {
public:

	/*!
	 * This is the standard constructor for the FileHandler. A specific one is not necessary
	 * because the FileHandler doesn't have any parameters.
	 * @return nothing
	 */
	FileHandler();
	/*!
	 * This is the standard copy constructor of an object.
	 * @return nothing to return
	 */
	FileHandler(const FileHandler &right);
	/*!
	 * This is the standard destructor of an object.
	 * @return nothing to return
	 */
	~FileHandler();
	/*!
	 * This is the standard assignment operator of an object of type FileHandler.
	 * @return the assigned FileHandler
	 */
	const FileHandler& operator=(const FileHandler &right);

	/*!
	 * This method reads the settings defined in the file named "settings". Different settings for one
	 * run is not possible, yet.
	 * @param settings This refers to the settings object which is used all over the program.
	 * @param settingsPath This is the path where the settings file can be found. Can be an empty string and can be defined via command line argument on program execution.
	 * @return True if all settings have been read.
	 */
	virtual bool readSettings(SetupConstants* settings, char* settingsPath);

	/*!
	 * This method reads out all eigenvalues and -vectors for ONE protein and ONE model except the first one (GNM) and accordingly the first six (ANM).
	 * @param path This is the path to the current [proteinName]_eigValsAndVecs[model].out file.
	 * @param protName The name of the protein which currently shall be analyzed.
	 * @param model This is the network model for which the calculated eigenvalues and -vectors are to be read in.
	 * @param setMap This refers to the settings map object which is used all over the program.
	 * @return A vector of vectors of type double. Every vector contains the eigenvalue as first element. The eigenvectors follow.
	 */
	virtual vector<vector<double> > readEigenvaluesAndVecs(const char* path, const char* protName, const char* model, SettingsMap &setMap, MassMap &allMasses);

	/*!
	 * This method reads only information to identify c_alpha atoms within the program. It writes the information into "settings".
	 * @param pqrmPath This is the path were the pqrm file of the protein can be found. Can be an empty string and can be defined in the settings file.
	 * @param protName This is the name of the protein which has to be processed. This name must NOT be an empty string. It the pqrm file name must be the same.
	 * @param settings This refers to the settings object which is used all over the program.
	 */
	virtual void readPQRM(const char* pqrmPath, const char* protName, BBMap &bbmap);

	/*!
	 * This method prints the calculated covariances to a file named [proteinName]_covs[model].out. The file structure is
	 * a table with three columns. The first two tell which relation is defined by the covariance, thus by the third column.
	 * @param covariances This is the matrix of covariances column by column.
	 * @param protName This is the name of the protein the analysis is done for.
	 * @param model This is the type of the network model.
	 * @param matrixDimensions This is the number of atoms the protein has. The covariance matrix has the dimensions numberOfAtoms x numberOfAtoms.
	 */
	virtual void printCovariances(double* covariances, const char* protName, const char* path, const char* model, int matrixDimensions);

	/*!
	 * This method prints the calculated correlations to a file named [proteinName]_corrs[model].out. The file structure is
	 * a table with three columns. The first two tell which relation is defined by the correlation, which you can find in the third column.
	 * @param correlations This is the matrix of correlations column by column.
	 * @param protName This is the name of the protein the analysis is done for.
	 * @param model This is the type of the network model.
	 * @param matrixDimensions This is the number of atoms the protein has. The correlation matrix has the dimensions numberOfAtoms x numberOfAtoms.
	 */
	virtual void printCorrelations(double* correlations, const char* protName, const char* path, const char* model, int matrixDimensions);

	/*!
	 * This method prints the calculated b values to a file named [proteinName]_bVals[model].out. The file structure is a
	 * table with two columns with the index of the atom in the first and the b value for this atom in the second column.
	 * @param bvalues This is the array of b values for all chosen atoms of the protein.
	 * @param protName This is the name of the protein the analysis is done for.
	 * @param model This is the type of the network model.
	 */
	virtual void printBValues(vector<double> bvalues, vector<int> bbNumbers, const char* protName, const char* path, const char* model);
};

#endif // FILEHANDLER_H
