/*---------------------------------------------------------
    This file is part of the program suite TEN
    (Tools for Elastic Networks)

    Copyright (C)

	Lars Ackermann
    G. Matthias Ullmann
    Bayreuth 2014

    www.bisb.uni-bayreuth.de

    This program is free software: you can redistribute
    it and/or modify it under the terms of the
    GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will
    be useful, but WITHOUT ANY WARRANTY; without even the
    implied warranty of MERCHANTABILITY or FITNESS FOR A
    PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the
    GNU Affero General Public License along with this
    program.  If not, see <http://www.gnu.org/licenses/>.
-----------------------------------------------------------*/


#include "FileHandler.h"
#include <string.h>
#include <fstream>
#include <iostream>
#include "SetupConstants.h"
#include <list>
#include <algorithm>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/split.hpp>
#include <vector>
#include "stdio.h"

using namespace std;
using namespace boost::algorithm;

typedef vector<string> split_string_vector_type;

FileHandler::FileHandler() {

}

FileHandler::FileHandler(const FileHandler &right) {
}

FileHandler::~FileHandler() {
}

const FileHandler& FileHandler::operator=(const FileHandler &right) {
	// handle self assignment
	return *this;
}

bool FileHandler::readSettings(SetupConstants* settings, char* settingsPath) {
	SettingsMap setMap = settings->getSettings();
	string delim = " ";
	string separator = ",";
	string line;
	ifstream settingsFile(settingsPath);
	if (settingsFile.is_open()) {
		while (settingsFile.good()) {
			getline(settingsFile, line);
			char writable[line.size() + 1];
			std::copy(line.begin(), line.end(), writable);
			writable[line.size()] = '\0';
			string::size_type loc = line.find(delim, 0);
			//If the initial two characters are '//' then continue because '//' is for comments
			if (loc != string::npos && strncmp(line.c_str(), "//", 2) != 0) {
				string key = line.substr(0, loc);
				remove(key.begin(), key.end(), ' ');
				string value = line.substr(loc + 1);
				if (settings->isKeyValid(key) and strlen(value.c_str()) != 0) {
					remove(value.begin(), value.end(), ' ');
					double doubleValue = atof(value.c_str());
					setMap[key] = doubleValue;
				} else if (strncmp(key.c_str(), "names", 5) == 0) {
					// split string for each separator
					split_string_vector_type splitted; // #2: Search for tokens
					split(splitted, value, is_any_of(separator), token_compress_on);
					settings -> setNames(splitted);
					//if separator found split string for each separator
				} else if (strncmp(key.c_str(), "protPaths", 9) == 0) {
					split_string_vector_type splitted; // #2: Search for tokens
					split(splitted, value, is_any_of(separator), token_compress_on);
					settings -> setPaths(splitted);
				} else if (strncmp(key.c_str(), "models", 6) == 0) {
					split_string_vector_type splitted; // #2: Search for tokens
					split(splitted, value, is_any_of(separator), token_compress_on);
					//The eigenvalues and -vectors for GNM are calculated if no model was given in the settings!
					if (splitted.empty()) {
						splitted.push_back("GNM");
					}
					settings -> setModels(splitted);
				} else {
					continue;
				}
			} else {
				continue;
			}
		}
		settings -> updateSettingsMap(setMap);
	} else {
		perror("Unable to open settings file. Please check the path you gave the program as first argument.");
		exit(1);
	}
	return true;
}

vector<vector<double> > FileHandler::readEigenvaluesAndVecs(const char* path, const char* protName, const char* model, SettingsMap &setMap, MassMap &allMasses) {

	//create file path
	const char* fileName = "_eigValsAndVecs";
	const char* fileNameExtension = ".out";
	char* fullFileName = new char[strlen(path) + strlen(protName) + strlen(model) + strlen(fileName) + strlen(fileNameExtension) + 1];
	strcpy(fullFileName, path);
	strcat(fullFileName, protName);
	strcat(fullFileName, fileName);
	strcat(fullFileName, model);
	strcat(fullFileName, fileNameExtension);

	string line;
	string delim = " ";
	string separator = ",";
	int start = 0;
	if (strncmp(model, "GNM", 3) == 0) {
		start = 1;
	} else if (strncmp(model, "ANM", 3) == 0) {
		start = 6;
	}

	vector<vector<double> > complete;

	ifstream eigValFile(fullFileName);
	if (eigValFile.is_open()) {
		int counter = 0;
		vector<double> currentEVV;
		while (eigValFile.good()) {
			getline(eigValFile, line);
			double currEV = 0.0;
			int brackPos = line.find("(");
			string massWHKey = "massWeightedHessian: ";
			int massWeightedHessianPos = line.find(massWHKey);
			string atomMassesKey = "Atom masses: ";
			int atomMassesPos = line.find(atomMassesKey);
			if (strncmp(line.c_str(), "EVal", 4) == 0) {
				if (counter >= start) {
					int pos = line.rfind(delim);
					string value = line.substr(pos + 1);
					currEV = atof(value.c_str());
					currentEVV.push_back(currEV);
				} else {
					counter++;
					continue;
				}
			} else if (brackPos != -1 and currentEVV.size() > 0) {
				int size = line.rfind(")");
				string values = line.substr(brackPos + 1, size - 2);
				split_string_vector_type splitted;
				split(splitted, values, is_any_of(separator), token_compress_on);
				for (uint i = 0; i < splitted.size(); i++) {
					string current = splitted.at(i);
					currentEVV.push_back(atof(current.c_str()));
				}
				complete.push_back(currentEVV);
				currentEVV.erase(currentEVV.begin(), currentEVV.end());
			} else if (massWeightedHessianPos != -1 and strncmp(model, "ANM", 3) == 0) {
				string value = line.substr(massWeightedHessianPos + strlen(massWHKey.c_str()), 1);
				setMap["massWeightedHessian"] = atof(value.c_str());
			} else if (atomMassesPos != -1 and strncmp(model, "ANM", 3) == 0) {
				string values = line.substr(atomMassesPos + strlen(atomMassesKey.c_str()));
				vector<double> masses;
				split_string_vector_type splitted;
				split(splitted, values, is_any_of(separator), token_compress_on);
				for (uint i = 0; i < splitted.size(); i++) {
					string cleanedValue = splitted.at(i);
					remove(cleanedValue.begin(), cleanedValue.end(), ' ');
					masses.push_back(atof(cleanedValue.c_str()));
				}
				allMasses[protName] = masses;
			}
		}
	} else {
		perror("Unable to open eigenvalue and -vector file. Please check the paths.");
		exit(1);
	}
	delete[] fullFileName;
	return complete;
}

void FileHandler::readPQRM(const char* pqrmPath, const char* protName, BBMap &bbmap) {
	vector<int> bbNumbers;
	list<string> temp;
	string line;

	char* fullPath = new char[strlen(pqrmPath) + strlen(protName) + strlen(".pqrm") + 1];
	fullPath = strcpy(fullPath, pqrmPath);
	fullPath = strcat(fullPath, protName);
	fullPath = strcat(fullPath, ".pqrm");

	ifstream pqrmFile(fullPath);
	if (pqrmFile.is_open()) {
		while (pqrmFile.good()) {
			getline(pqrmFile, line);
			char *writable = new char[line.size() + 1];
			std::copy(line.begin(), line.end(), writable);
			writable[line.size()] = '\0';
			if (strncmp(writable, "ATOM  ", 6) == 0 or strncmp(writable, "HETATM  ", 8) == 0) {
				temp.push_back(line);
			} else {
				delete[] writable;
				continue;
			}
			delete[] writable;
		}

		int i = 0;
		char *atom_type = new char[5];

		while (!temp.empty()) {
			string currentLineString = temp.front();
			char *writable = new char[currentLineString.size() + 1];
			std::copy(currentLineString.begin(), currentLineString.end(), writable);
			writable[currentLineString.size()] = '\0';
			sscanf(writable, "%*s %*s %s %*s %*d %*lf %*lf %*lf %*s %*s %*lf\n", atom_type);
			if (strncmp(atom_type, "BB", 2) == 0) {
				bbNumbers.push_back(i);
			}
			temp.pop_front();
			i++;
			delete[] writable;
		}
		bbmap[protName] = bbNumbers;
		pqrmFile.close();
		delete[] fullPath;
		delete[] atom_type;
	} else {
		delete[] fullPath;
		perror("Unable to open pqrm file. Check file path to setup file and the configured protein names.");
		exit(1);
	}
}

void FileHandler::printCovariances(double* covariances, const char* protName, const char* path, const char* model, int matrixDimensions) {
	//create file path
	const char* fileName = "_covs";
	const char* fileNameExtension = ".out";
	char* fullFileName = new char[strlen(path) + strlen(protName) + strlen(model) + strlen(fileName) + strlen(fileNameExtension) + 1];
	strcpy(fullFileName, path);
	strcat(fullFileName, protName);
	strcat(fullFileName, fileName);
	strcat(fullFileName, model);
	strcat(fullFileName, fileNameExtension);

	ifstream eigValVecOutputFileToDelete(fullFileName);
	if (eigValVecOutputFileToDelete) {
		eigValVecOutputFileToDelete.close();
		remove(fullFileName);
	}
	FILE* eigValVecOutputFile;
	/* open the file*/
	if ((eigValVecOutputFile = fopen(fullFileName, "a+")) == NULL) {
		perror("Could not create and/or open the file with the covariances. Make sure you have closed it and you have the rights for writing within their directory.");
		exit(1);
	} else {
		for (int i = 0; i < matrixDimensions; i++) {
			for (int j = 0; j < matrixDimensions; j++) {
				fprintf(eigValVecOutputFile, "%d %d %g\n", i + 1, j + 1, covariances[i * matrixDimensions + j]);
			}
		}
	}
	delete[] fullFileName;
}

void FileHandler::printCorrelations(double* correlations, const char* protName, const char* path, const char* model, int matrixDimensions) {
	//create file path
	const char* fileName = "_corrs";
	const char* fileNameExtension = ".out";
	char* fullFileName = new char[strlen(path) + strlen(protName) + strlen(model) + strlen(fileName) + strlen(fileNameExtension) + 1];
	strcpy(fullFileName, path);
	strcat(fullFileName, protName);
	strcat(fullFileName, fileName);
	strcat(fullFileName, model);
	strcat(fullFileName, fileNameExtension);

	ifstream eigValVecOutputFileToDelete(fullFileName);
	if (eigValVecOutputFileToDelete) {
		eigValVecOutputFileToDelete.close();
		remove(fullFileName);
	}
	FILE* eigValVecOutputFile;
	/* open the file*/
	if ((eigValVecOutputFile = fopen(fullFileName, "a+")) == NULL) {
		perror("Could not create and/or open the file with the correlations. Make sure you have closed it and you have the rights for writing within their directory.");
		exit(1);
	} else {
		for (int i = 0; i < matrixDimensions; i++) {
			for (int j = 0; j < matrixDimensions; j++) {
				fprintf(eigValVecOutputFile, "%d %d %g\n", i + 1, j + 1, correlations[i * matrixDimensions + j]);
			}
		}
	}
	delete[] fullFileName;
}

void FileHandler::printBValues(vector<double> bvalues, vector<int> bbNumbers, const char* protName, const char* path, const char* model) {
	//create file path
	const char* fileName = "_bVals";
	const char* fileNameExtension = ".out";
	char* fullFileName = new char[strlen(path) + strlen(protName) + strlen(model) + strlen(fileName) + strlen(fileNameExtension) + 1];
	strcpy(fullFileName, path);
	strcat(fullFileName, protName);
	strcat(fullFileName, fileName);
	strcat(fullFileName, model);
	strcat(fullFileName, fileNameExtension);

	ifstream eigValVecOutputFileToDelete(fullFileName);
	if (eigValVecOutputFileToDelete) {
		eigValVecOutputFileToDelete.close();
		remove(fullFileName);
	}
	FILE* bValOutputFile;
	/* open the file*/
	if ((bValOutputFile = fopen(fullFileName, "a+")) == NULL) {
		perror("Could not create and/or open the file with the B-values. Make sure you have closed it and you have the rights for writing within their directory.");
		exit(1);
	} else {
		if (bbNumbers.size() == bvalues.size()) {
			for (uint i = 0; i < bvalues.size(); i++) {
				fprintf(bValOutputFile, "%d %g\n", bbNumbers.at(i)+1, bvalues[i]);
			}
		} else {
			for (uint i = 0; i < bvalues.size(); i++) {
				fprintf(bValOutputFile, "%d %g\n", i+1, bvalues[i]);
			}
		}
	}
	delete[] fullFileName;
}
