/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @date
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The AbstractPseudoInvCalculator class is an abstract class which declares which functions a pseudo inverse calculator must have.
 * DO NOT create an instance of this class. Extend this abstract class to implement a valid calculator.
 */

#ifndef ABSTRACTPSEUDOINVCALCULATOR_H_
#define ABSTRACTPSEUDOINVCALCULATOR_H_

#include "SetupConstants.h"
#include "FileHandler.h"

/*
 *
 */
class AbstractPseudoInvCalculator {
public:
	/*!
	 * This is the constructor for the AbstractPseudoInvCalculator. DO NOT USE IT! If you use it, the compiler will throw an error.
	 * Instead using this constructor create an instance of a child class e.g. \link<PseudoInvCalculator.h>PseudoInvCalculator\endlink
	 * @param settings These are the setup options you defined in your settings file.
	 * @param fileHandler This is an instance of an i/o manager.
	 * @return nothing to return
	 */
	AbstractPseudoInvCalculator(SetupConstants* settings, FileHandler* fileHandler);

	/*!
	 * This is the standard destructor.
	 * @return nothing to return
	 */
	virtual ~AbstractPseudoInvCalculator();

	/*!
	 * This method calculates the pseudo inverse, the covariances, the correlations and the theoretical b values. Then it uses a fileHandler
	 * object to print the results.
	 * @param eigenvals These are the eigenvalues calculated from the kirchhoff matrix (GNM) or the hessian matrix (ANM).
	 * @param eigenvecs These are the eigenvectors calculated from the kirchhoff matrix (GNM) or the hessian matrix (ANM).
	 * @param protName This is the name of the protein the analysis is done for.
	 * @param model This is the type of the network model the eigenvalues and eigenvectors belong to.
	 * @param matrixDimensions This is the number of atoms (GNM) or 3 x the number of atoms (ANM).
	 */
	virtual void analyzeData(double* eigenvals, double* eigenvecs, const char* protName, const char* path, const char* model, int matrixDimensions) = 0;

	/*!
	 * The hessian matrix may be multiplied by M^-1/2 (2 times) which means that the hessian elements are weighted with the atom masses.
	 * In order to calculate the pseudo inverse you have to recalculate the modified hessian matrix. For more information please have a look
	 * at the user manual.
	 * @param hessianMatrix This is the mass weighted hessian matrix.
	 * @param lengthOfEVecs This is the number of atoms (GNM) or 3 x the number of atoms (ANM).
	 * @return The hessian matrix with no orthogonal eigenvectors but instead of that correct mass weighted values. For more information please
	 * have a look at the user manual.
	 */
	virtual double* prepareHessianMatrix(double* hessianMatrix, int lengthOfEVecs, const char* protName) = 0;

	/*!
	 * This method calculates the pseudo inverse for a matrix by using eigenvalues and -vectors.
	 * @param eigenvals These are the eigenvalues calculated from the kirchhoff matrix (GNM) or the hessian matrix (ANM).
	 * @param eigenvecs These are the eigenvectors calculated from the kirchhoff matrix (GNM) or the hessian matrix (ANM).
	 * @param model This is the type of the network model the eigenvalues and eigenvectors belong to.
	 * @param lengthOfEVecs This is the number of atoms (GNM) or 3 x the number of atoms (ANM).
	 * @return
	 */
	virtual double* calculatePseudoInverse(double* eigenvals, double* eigenvecs, const char* model, int lengthOfEVecs) = 0;

	/*!
	 * This method calculates the covariances using a pseudo inverse.
	 * @param invMatrix This is the pseudo inverse whether of the kirchhoff matrix or the hessian matrix.
	 * @param lengthOfEVecs This is the number of atoms (GNM) or 3 x the number of atoms (ANM).
	 * @param model
	 * @return The covariance matrix for the pseudo inverse.
	 */
	virtual double* calculateCovariances(double* invMatrix, int lengthOfEVecs, const char* model) = 0;

	/*!
	 * This method calculates the correlations using the covariance matrix. The values of the correlation matrix are
	 * standardized. So it would be strange if any value of the result correlation matrix is more than +1 or less than -1.
	 * @param covarianceMatrix This is the covariance matrix of the pseudo inverse.
	 * @param lengthOfEVecs This is the number of atoms the protein has.
	 * @return The correlation matrix.
	 */
	virtual double* calculateCorrelations(double* covarianceMatrix, int lengthOfEVecs) = 0;

	/*!
	 * This method calculates the theoretical b values using the covariance matrix.
	 * @param covarianceMatrix This is the covariance matrix of the pseudo inverse.
	 * @param lengthOfEVecs This is the number of atoms the protein has.
	 * @return The theroretical b values.
	 */
	virtual vector<double> calculateTheoreticalBValues(double* covarianceMatrix, int lengthOfEVecs, const char* protName) = 0;

protected:
	SetupConstants* settings;
	FileHandler* fileHandler;
};

#endif /* ABSTRACTPSEUDOINVCALCULATOR_H_ */
