/*---------------------------------------------------------
    This file is part of the program suite TEN
    (Tools for Elastic Networks)

    Copyright (C)

	Lars Ackermann
    G. Matthias Ullmann
    Bayreuth 2014

    www.bisb.uni-bayreuth.de

    This program is free software: you can redistribute
    it and/or modify it under the terms of the
    GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will
    be useful, but WITHOUT ANY WARRANTY; without even the
    implied warranty of MERCHANTABILITY or FITNESS FOR A
    PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the
    GNU Affero General Public License along with this
    program.  If not, see <http://www.gnu.org/licenses/>.
-----------------------------------------------------------*/


#include "main.h"
#include <iostream>
#include "FileHandler.h"
#include "SetupConstants.h"
#include "TrajectoryGeneratorLapack.h"
#include <stdlib.h>
#include <string.h>
#include <vector>
#include <stdio.h>
#include <time.h>

using namespace std;

/*possible arguments in argv:
 * [0] - contains the command itself, which means the location of the executed program.
 * [1] - absolute path (including file name) to the setup file to use (there it is specified which proteins shall be processed [names ...], where they are [paths...]...)
 *
 *settings:
 *	protNames: names of proteins
 *	pqrmPaths: paths for files of the proteins for all protNames (only folder path; NO FILE NAME AT THE END)
 *
 * */
int main(int argc, char **argv) {

	clock_t init, final,allPrevious;
	init = clock();

	char* setupFilePath;
	if (argc < 2) {
		setupFilePath = "settings";
	} else {
		setupFilePath = argv[1];
	}

	SetupConstants* constants = new SetupConstants();
	FileHandler* fileHandler = new FileHandler();
	fileHandler->readSettings(constants, setupFilePath);
	SettingsMap settings = constants->getSettings();
	if(settings["print_traj"] == 0 and settings["print_super"] == 0 and settings["compareConfs"] == 0){
		std::cout << "WARNING: You instructed the program to run but the settings 'print_traj', 'print_super', 'compareConfs' are all zero. So the program would not provide any result. Program stops now." << std::endl;
		exit(0);
	}
	vector<string> protPaths = constants -> getPaths();
	MassMap allMasses = constants -> getMasses();
	BBMap allBBNumbers = constants -> getBBNumbers();

	vector<string> names = constants -> getNames();
	vector<string> refNames = constants -> getRefNames();
	if (refNames.size() == 0 and settings["compareConfs"] == 1) {
		std::cout
				<< "WARNING: You instructed the program to compare your working conformations to a reference conformation. To be able to compute this comparison, please specify the name of the reference conformation with the setting 'refNames'.\n This time the comparison will be ignored."
				<< std::endl;
		settings["compareConfs"] = 0;
		constants -> updateSettingsMap(settings);
	}

	if(refNames.size() != names.size() and settings["compareConfs"] == 1){
		std::cout << "WARNING: You instructed the program to compare your working conformations to a reference conformation but the number of reference structure names and protein names are not equal. Program will only compare the first " << refNames.size() << " structure(s)." << std::endl;
	}

	int compConfs = settings["compareConfs"];
	int trajModesVectorSize = (constants -> getTrajs()).size();
	int superTrajModesVectorSize = (constants -> getSuperTrajs()).size();
	for (uint i = 0; i < (names.size()); i++) {
		allPrevious = clock() - init;
		int currentTMVS = (constants -> getTrajs()).size();
		int currentSTMVS = (constants -> getSuperTrajs()).size();
		//if more than one protein is processed we need to reset the options which must be changed because of wrong settings (settings file). Some of these settings depend on the protein.
		if(trajModesVectorSize != currentTMVS or superTrajModesVectorSize != currentSTMVS){
			fileHandler -> readSettings(constants, setupFilePath);
			settings = constants -> getSettings();
			settings["compareConfs"] = compConfs;
			std::cout << "WARNING: Due to fact that either the 'traj_modes' or 'supertraj_modes' option or even both were out of range for the protein the user settings must be read again. Correct the settings to improve performance." << std::endl;
		}
		if(settings["compareConfs"] == 1 and i >= refNames.size()){
			settings["compareConfs"] = 0;
			constants -> updateSettingsMap(settings);
		}
		const char* currentName = names.at(i).c_str();
		const char* currentModel = "ANM";
		string currentPath;
		if (i < protPaths.size()) {
			currentPath = protPaths.at(i);
			int pathLength = strlen(currentPath.c_str());
			char ending = currentPath.at(pathLength - 1);
			if (ending != '/') {
				currentPath.append("/");
			}
		} else {
			currentPath = "";
		}

		std::cout << i + 1 << ".: Processing protein " << currentName << ":" << std::endl;
		std::cout << "Reading pqrm to load atom start coordinates...";
		vector<vector<double> > startCoords = fileHandler->readPQRM(currentPath.c_str(), currentName, allBBNumbers);
		std::cout << "done" << std::endl;
		std::cout << "Reading eigenvalues and -vectors...";
		vector<vector<double> > evvcs = fileHandler -> readEigenvaluesAndVecs(currentPath.c_str(), currentName, currentModel, settings, allMasses);
		std::cout << "done" << std::endl;
		constants -> updateSettingsMap(settings);
		constants -> setAllMasses(allMasses);
		constants -> setAllBBNumbers(allBBNumbers);

		vector<double> eigenvals;
		//the dimension has to be (first.size()-1)^2 because the first element is the eigenvalue and must be ignored for calculating the size of the eigenvector array.
		vector<double> first = evvcs.at(0);

		double* eigenvecs = new double[(first.size() - 1) * (first.size() - 1)];
		for (uint k = 0; k < evvcs.size(); k++) {
			vector<double> evvc = evvcs.at(k);
			//the first element of evvc is the eigenvalue
			eigenvals.push_back(evvc.at(0));
			for (uint l = 1; l < first.size(); l++) {
				eigenvecs[k * (first.size() - 1) + l - 1] = evvc.at(l);
			}
		}

		vector<int> trajs = constants -> getTrajs();
		//trajs is sorted so i can simply check where the first non-existing traj is and delete from there to the end of the vector.
		for (uint j = 0; j < trajs.size(); j++) {
			uint traj = trajs.at(j);
			if (traj > eigenvals.size()) {
				std::cout << "WARNING: You wanted to print trajectories which cannot exist (single trajectory). The last trajectory has the index " << eigenvals.size()
						<< ". The program drops the trajectory no. " << traj << " as well as each one which has an index even greater." << std::endl;
				trajs.erase(trajs.begin() + j, trajs.end());
				break;
			}
		}
		constants -> setTrajs(trajs);
		if (settings["print_traj"] != 0 and trajs.size() == 0) {
			std::cout << "WARNING: You want to print single trajectories but forgot to specify which ones. The program will not print any single trajectory." << std::endl;
		}

		//supertrajs is sorted so i can simply check where the first non-existing traj is and delete from there to the end of the vector.
		vector<int> supertrajs = constants -> getSuperTrajs();
		for (uint j = 0; j < supertrajs.size(); j++) {
			uint traj = supertrajs.at(j);
			if (traj > eigenvals.size()) {
				std::cout << "WARNING: You wanted to print trajectories which cannot exist (super trajectory). The last trajectory has the index " << eigenvals.size()
						<< ". The program drops the trajectory no. " << traj << " as well as each one which has an index even greater." << std::endl;
				supertrajs.erase(supertrajs.begin() + j, supertrajs.end());
				break;
			}
		}
		constants -> setSuperTrajs(supertrajs);

		AbstractTrajectoryGenerator* trGen = new TrajectoryGeneratorLapack(constants, fileHandler);

		if (settings["massWeightedHessian"] == 1) {
			eigenvecs = trGen -> prepareHessianMatrix(eigenvecs, eigenvals.size(), currentName);
		}

		if ((settings["print_traj"] == 1 or settings["print_traj"] == 2 or settings["compareConfs"] == 1) and trajs.size() > 0) {
			std::cout << "Calculating " << trajs.size() << " trajectories (print mode: " << settings["print_traj"] << ")...";
			trGen -> calcAndPrintSingleTrajectories(currentName, startCoords, eigenvals, eigenvecs, i);
			std::cout << "done" << std::endl;
		}
		if (settings["print_super"] == 1 or settings["print_super"] == 2 or settings["compareConfs"] == 1) {
			std::cout << "Calculating super trajectory of " << supertrajs.size() << " modes (print mode: " << settings["print_super"] << ")...";
			trGen -> calcAndPrintSuperTrajectories(currentName, startCoords, eigenvals, eigenvecs, i);
			std::cout << "done" << std::endl;
		}
		delete trGen;
		delete[] eigenvecs;
		final = clock() - init - allPrevious;
		std::cout << "Time: "  << (double) final / ((double) CLOCKS_PER_SEC) << std::endl;
	}

	delete fileHandler;
	delete constants;

	std::cout << "All done... ";
	final = clock() - init;
	std::cout << "Total time: " << (double) final / ((double) CLOCKS_PER_SEC) << std::endl;

	return 0;
}
