/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @date
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The TrajectoryGeneratorLapack is an implementation of the abstract class \link<AbstractTrajectoryGenerator>AbstractTrajectoryGenerator\endlink . It provides functionalities to calculate trajectories basing on eigenvalues and -vectors.
 * It uses Lapack++ for some matrix operations. This in turn uses the original Fortran implementation of
 * Lapack and so Lapack++ is only a wrapper.
 */

#ifndef TRAJECTORYGENERATORLAPACK_H_
#define TRAJECTORYGENERATORLAPACK_H_

#include "AbstractTrajectoryGenerator.h"

/*
 *
 */
class TrajectoryGeneratorLapack: public AbstractTrajectoryGenerator {
public:
	/*!
	 * This is the constructor for TrajectoryGeneratorLapack which is a child class of \link<AbstractTrajectoryGenerator>AbstractTrajectoryGenerator\endlink . This constructor uses the super constructor. For more information please check the latter.
	 * @param settings
	 * @param fileHandler
	 * @return
	 */
	TrajectoryGeneratorLapack(SetupConstants *settings, FileHandler* fileHandler);

	/*!
	 * This is the standard destructor.
	 * @return nothing to return
	 */
	virtual ~TrajectoryGeneratorLapack();

	/*!
	 * This method calculates a single trajectory which means a vibration which bases upon ONE eigenvector/mode (specified via settings file). For the calculation Lapack++ is used.
	 * Furthermore the method prints this trajectory either as dcd file or as pdb file. For printing the method uses the fileHandler object.
	 * If configured the method also compares the protein conformation with a reference structure each time step. To be able to do this the program uses the \link<ConfComparator>ConfComparator\endlink .
	 * @param protName The name of the protein which is currently processed.
	 * @param startCoords The start coordinates for all relevant atoms which were read from the pqrm file.
	 * @param eigenvals The eigenvalues (3 times the number of atoms - 6)  of the hessian matrix. For more information please check the user manual.
	 * @param eigenvecs The eigenvectors (3 times the number of atoms - 6) of the hessian matrix. For more information please check the user manual.
	 * @param internalProtId The internal protein id which means the position of the protein name in the vector of protein names (see \link<main>main\endlink)
	 */
	virtual void calcAndPrintSingleTrajectories(const char* protName, vector<vector<double> > startCoords, vector<double> eigenvals, double* eigenvecs, int internalProtId);

	/*!
	 * This method calculates a super trajectory which means a vibration which bases upon all configured eigenvectors/modes (specified via settings file). For this calculation Lapack++ is used.
	 * Furthermore the method prints this super trajectory either as dcd file or as pdb file. For printing the method uses the fileHandler object.
	 * If configured the method also compares the protein conformation with a reference structure each time step. To be able to do this the program uses the \link<ConfComparator>ConfComparator\endlink .
	 * @param protName The name of the protein which is currently processed.
	 * @param startCoords The start coordinates for all relevant atoms which were read from the pqrm file.
	 * @param eigenvals The eigenvalues (3 times the number of atoms - 6)  of the hessian matrix. For more information please check the user manual.
	 * @param eigenvecs The eigenvectors (3 times the number of atoms - 6) of the hessian matrix. For more information please check the user manual.
	 * @param internalProtId The internal protein id which means the position of the protein name in the vector of protein names (see \link<main>main\endlink)
	 */
	virtual void calcAndPrintSuperTrajectories(const char* protName, vector<vector<double> > startCoords, vector<double> eigenvals, double* eigenvecs, int internalProtId);

	/*!
	 * The hessian matrix may be multiplied by M^-1/2 (2 times) which means that the hessian elements are weighted with the atom masses.
	 * In order to calculate the trajectories you have to recalculate the modified hessian matrix. For more information please have a look
	 * at the user manual.
	 * @param hessianMatrix This is the mass weighted hessian matrix.
	 * @param numberOfEVals This is 3 x the number of atoms - 6 (because the first 6 eigenvalues are zero).
	 * @return The hessian matrix with no orthogonal eigenvectors but instead of that correct mass weighted values. For more information please
	 * have a look at the user manual.
	 */
	virtual double* prepareHessianMatrix(double* modifiedHessianMatrix, int lengthOfEVecs, const char* protName);

	/*!
	 * The task of this method is to calculate a random value between -1 and +1 for all not manually configured phases.
	 * @param phaseMap The manually configured phases mapped to their modes.
	 * @param superTrajs Contains the indices of modes which are to be used for computing the super trajectory.
	 * @return
	 */
	virtual PhaseMap preparePhasesMap(PhaseMap phaseMap, vector<int> superTrajs);

private:
	virtual float** calculateNewCoords(vector<vector<double> > startCoords, double* displacementStep);
	virtual double* calculateSingleDisplacementStep(vector<double> linExpansions, double* eigenvecs, int trajIndex, double time);
	virtual double* calculateSuperDisplacementStep(vector<double> linExpansions, double* eigenvec, vector<double> eigenvals, int trajIndex, double time, PhaseMap phases);
	virtual vector<double> calculateLinearExpansions(vector<double> eigenvalues);
};

#endif /* TRAJECTORYGENERATOR_H_ */
