/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @date
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The FileHandler class is responsible for input and output actions. It reads the
 * user's settings file, the files with the eigenvalues and -vectors and the pqrm file.
 * As possible output formats there are .dcd and .pdb files as well as a .dist file. DCD is a binary file which has a special format,
 * defined on the following web site: http://www.ks.uiuc.edu/Research/vmd/plugins/molfile/dcdplugin.html. The pdb file is only a pseudo pdb since it only consists of the coordinates of the atoms for each time step.
 */

#ifndef FILEHANDLER_H
#define FILEHANDLER_H
#include "SetupConstants.h"
#include <utility>

/* CHARMm trajectory file header */
typedef struct {
	/* First Record */
	char type[6];
	int info[20];
	/* Second Record */
	int no_title;
	char **title;
	/* Third record */
	int no_atoms;
} TrajectHeader;

/* constant definitions for 'info'-array */
#define N_SET	    0	/* Number of data sets				*/
#define N_START     1	/* Number of dynamic steps, which has been per-      */
/* formed before the trajectory has been written out */
#define INTERV	    2	/* intervall in which the coordinates have been */
/* save						*/
#define N_STEPS	    3	/* Number of dynamic steps, protocolled in the	*/
/* actual trajectory				*/
#define N_DOF	    7	/* Number of the degrees of freedom		*/
#define N_FIX	    8   /* Number of fixed atoms			*/
#define SEED	    9	/* Seed for the random number generator		*/
#define PRES	   10	/* Flag: 1 for crystal/constant pressure calc., */
/*	 0 otherwise    			*/
#define CHARMM_VER 19	/* CHARMm Version number			*/

class FileHandler {
public:

	/*!
	 * This is the standard constructor for the FileHandler. A specific one is not necessary
	 * because the FileHandler doesn't have any parameters.
	 * @return nothing
	 */
	FileHandler();
	/*!
	 * This is the standard copy constructor of an object.
	 * @return nothing to return
	 */
	FileHandler(const FileHandler &right);
	/*!
	 * This is the standard destructor of an object.
	 * @return nothing to return
	 */
	~FileHandler();
	/*!
	 * This is the standard assignment operator of an object of type FileHandler.
	 * @return the assigned FileHandler
	 */
	const FileHandler& operator=(const FileHandler &right);

	/*!
	 * This method reads the settings defined in the file named "settings". Different settings for one
	 * run is not possible, yet.
	 * @param settings This refers to the settings object which is used all over the program.
	 * @param settingsPath This is the path where the settings file can be found. Can be an empty string and can be defined via command line argument on program execution.
	 * @return True if all settings have been read.
	 */
	virtual bool readSettings(SetupConstants* settings, char* settingsPath);

	/*!
	 * This method reads atom coordiantes from the pqrm file and holds it in a two dimensional vector. It also collects hints which atoms are members of the back bone of the protein structure. This collection is saved in the already created SetupConstants object.
	 * @param pqrmPath This is the path were the pqrm file of the protein can be found. Can be an empty string and can be defined in the settings file.
	 * @param protName This is the name of the protein which has to be processed. This name must NOT be an empty string. It the pqrm file name must be the same.
	 * @param settings This refers to the settings object which is used all over the program.
	 * @return the coordinates as a two dimensional vector (N x 3).
	 */
	virtual vector<vector<double> > readPQRM(const char* pqrmPath, const char* protName, BBMap &bbmap);

	/*!
	 * This method reads out all eigenvalues and -vectors for ONE protein and ONE model except the first six (ANM).
	 * @param path This is the path to the current [proteinName]_eigValsAndVecs[model].out file.
	 * @param protName The name of the protein which currently shall be processed.
	 * @param model This is the network model for which the calculated eigenvalues and -vectors are to be read in (always ANM in this program).
	 * @param setMap This refers to the settings map object which is used all over the program.
	 * @return A vector of vectors of type double. Every vector contains the eigenvalue as first element. The eigenvectors follow.
	 */
	virtual vector<vector<double> > readEigenvaluesAndVecs(const char* path, const char* protName, const char* model, SettingsMap &setMap, MassMap &allMasses);

	/*!
	 * This method creates and opens a new DCD file.
	 * @param modeIndex This is the index of the eigenvector (but the index+1 of the eigenvector internally).
	 * @param protName This is the protein the DCD file is to be belong to.
	 * @return the open DCD file
	 */
	virtual FILE* createAndOpenDCDFile(int modeIndex, const char* protName, const char* path);

	/*!
	 * This method creates and opens a new PDB file.
	 * @param modeIndex This is the index of the eigenvector (but the index+1 of the eigenvector internally).
	 * @param protName This is the protein the PDB file is to be belong to.
	 * @return the open PDB file
	 */
	virtual FILE* createAndOpenPDBFile(int modeIndex, const char* protName, const char* path);

	/*!
	 * This method creates and opens a new Dist file.
	 * @param modeIndex This is the index of the eigenvector (but the index+1 of the eigenvector internally).
	 * @param protName This is the protein the Dist file is to be belong to.
	 * @return the open Dist file
	 */
	virtual FILE* createAndOpenDistFile(int modeIndex, const char* protName, const char* path);

	/*!
	 * This method writes the required standard header of dcd files. For more information please have a look at: http://www.ks.uiuc.edu/Research/vmd/plugins/molfile/dcdplugin.html
	 * @param dcdFile The DCD file for the current protein.
	 * @param natom The number of atoms the protein has.
	 * @param trajSteps The number of steps (frames) the movie shall have.
	 */
	virtual void writeDCDHeader(FILE* dcdFile, int natom, int trajSteps);

	/*!
	 * Writes one data block of new coordinates according to the DCD format. For more information about the format, please have a look at: http://www.ks.uiuc.edu/Research/vmd/plugins/molfile/dcdplugin.html
	 * @param dcdFile The DCD file for the current protein.
	 * @param coords The coordinates for the current time step. Each row of the two dimensional array contains coordinates for one axis of coordinates.
	 * @param natoms The number of atoms the protein has.
	 */
	virtual void writeDCD(FILE* dcdFile, float** coords, int natoms);

	/*!
	 * Writes the coordinates for the current time step to the pseudo PDB file.
	 * @param pdbFile The PDB file for the current protein.
	 * @param coords The coordinates for the current time step. Each row of the two dimensional array contains coordinates for one axis of coordinates.
	 * @param natom The number of atoms the protein has.
	 */
	virtual void writePseudoPDB(FILE* pdbFile, float** coords, int natom);

	/*!
	 * Writes the configured number of distances with the time steps to the Dist file.
	 * @param file The .dist file for the current protein.
	 * @param dists contains pairs of time step and distance.
	 */
	virtual void writeDists(FILE* file, vector< pair<double,float> > dists);

	/*!
	 * Closes a file with fclose(FILE* file) and provides an error message if anything fails.
	 * @param file The file to close.
	 */
	virtual void closeFile(FILE* file);

private:
	virtual void saveWriteBinary(const void * ptr, int size, uint number, FILE* output);
	virtual void generateCHARMmHeader(TrajectHeader *header, long seed, int natom, int n_start, int interv, int n_steps);
	virtual FILE* writeCHARMmHeader(TrajectHeader* header, FILE* dcdFile);
	virtual void fprint_header_info(FILE *fpout, TrajectHeader *header);
};

#endif // FILEHANDLER_H
