/*---------------------------------------------------------
    This file is part of the program suite TEN
    (Tools for Elastic Networks)

    Copyright (C)

	Lars Ackermann
    G. Matthias Ullmann
    Bayreuth 2014

    www.bisb.uni-bayreuth.de

    This program is free software: you can redistribute
    it and/or modify it under the terms of the
    GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will
    be useful, but WITHOUT ANY WARRANTY; without even the
    implied warranty of MERCHANTABILITY or FITNESS FOR A
    PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the
    GNU Affero General Public License along with this
    program.  If not, see <http://www.gnu.org/licenses/>.
-----------------------------------------------------------*/

#include "FileHandler.h"
#include <string.h>
#include <fstream>
#include <iostream>
#include <sstream>
#include "SetupConstants.h"
#include <list>
#include <algorithm>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/split.hpp>
#include <vector>
#include "stdio.h"

extern "C" {
#include "util.h"
}

using namespace std;
using namespace boost::algorithm;

typedef vector<string> split_string_vector_type;

FileHandler::FileHandler() {

}

FileHandler::FileHandler(const FileHandler &right) {
}

FileHandler::~FileHandler() {
}

const FileHandler& FileHandler::operator=(const FileHandler &right) {
	// handle self assignment
	return *this;
}

bool FileHandler::readSettings(SetupConstants* settings, char* settingsPath) {
	SettingsMap setMap = settings->getSettings();
	string delim = " ";
	string separator = ",";
	string line;
	ifstream settingsFile(settingsPath);
	if (settingsFile.is_open()) {
		while (settingsFile.good()) {
			getline(settingsFile, line);
			char *writable = new char[line.size() + 1];
			std::copy(line.begin(), line.end(), writable);
			writable[line.size()] = '\0';
			string::size_type loc = line.find(delim, 0);
			//If the initial two characters are '//' then continue because '//' is for comments
			if (loc != string::npos && strncmp(line.c_str(), "//", 2) != 0) {
				string key = line.substr(0, loc);
				const char* keyC = key.c_str();
				remove(key.begin(), key.end(), ' ');
				string value = line.substr(loc + 1);
				if (settings->isKeyValid(key) and strlen(value.c_str()) != 0) {
					remove(value.begin(), value.end(), ' ');
					string::size_type slashLoc = line.find("/", 0);
					double doubleValue = 0.0;
					if (slashLoc != string::npos) {
						split_string_vector_type splitted;
						split(splitted, value, is_any_of("/"), token_compress_on); // #2: Search for tokens
						double numerator = atof(splitted.at(0).c_str());
						double denominator = atof(splitted.at(1).c_str());
						doubleValue = numerator / denominator;
					} else {
						doubleValue = atof(value.c_str());
					}
					setMap[key] = doubleValue;
				} else if (strncmp(keyC, "names", 5) == 0) {
					// split string for each separator
					split_string_vector_type splitted;
					split(splitted, value, is_any_of(separator), token_compress_on); // #2: Search for tokens: if separator found split string for each separator
					settings -> setNames(splitted);
				} else if (strncmp(keyC, "protPaths", 9) == 0) {
					split_string_vector_type splitted;
					split(splitted, value, is_any_of(separator), token_compress_on); // #2: Search for tokens
					settings -> setPaths(splitted);
				}else if (strncmp(keyC, "refNames", 8) == 0) {
					// split string for each separator
					split_string_vector_type splitted;
					split(splitted, value, is_any_of(separator), token_compress_on); // #2: Search for tokens: if separator found split string for each separator
					settings -> setRefNames(splitted);
				} else if (strncmp(keyC, "refPaths", 8) == 0) {
					split_string_vector_type splitted;
					split(splitted, value, is_any_of(separator), token_compress_on); // #2: Search for tokens
					settings -> setRefPaths(splitted);
				} else if (strncmp(keyC, "phases", 6) == 0) {
					split_string_vector_type splitted;
					split(splitted, value, is_any_of(separator), token_compress_on); // #2: Search for tokens
					int lastSetPhase = 1;
					for (uint i = 0; i < splitted.size(); i++) {
						string currentToken = splitted.at(i);
						string::size_type brackOpLoc = currentToken.find("[", 0);
						//case: x[a] or x[a-b]
						if (brackOpLoc != string::npos) {
							string::size_type brackClLoc = currentToken.find("]", 0);
							string realValueString = currentToken.substr(0, brackOpLoc);
							double realValue = atof(realValueString.c_str());
							string::size_type hyphenLoc = currentToken.find("-", 0);
							//case: x[a-b]
							if (hyphenLoc != string::npos) {
								string bondsString = currentToken.substr(brackOpLoc + 1, brackClLoc - brackOpLoc - 1);
								split_string_vector_type bonds;
								split(bonds, bondsString, is_any_of("-"), token_compress_on);
								int lowerBond = atoi(bonds.at(0).c_str());
								int upperBond = atoi(bonds.at(1).c_str());
								for (int j = lowerBond; j <= upperBond; j++) {
									settings -> addToPhaseMap(j, realValue);
								}
								lastSetPhase = upperBond;
								//case: x[a]
							} else {
								string indexString = currentToken.substr(brackOpLoc + 1, brackClLoc - brackOpLoc - 1);
								int index = atoi(indexString.c_str());
								settings -> addToPhaseMap(index, realValue);
								lastSetPhase = index;
							}
							//case: x
						} else {
							lastSetPhase++;
							double realValue = atof(currentToken.c_str());
							settings -> addToPhaseMap(lastSetPhase, realValue);
						}
					}
				} else if (strncmp(keyC, "traj_modes", 10) == 0) {
					vector<int> trajs;
					split_string_vector_type splitted;
					split(splitted, value, is_any_of(separator), token_compress_on); // #2: Search for tokens
					for (uint i = 0; i < splitted.size(); i++) {
						string currentToken = splitted.at(i);
						string::size_type hyphenLoc = currentToken.find("-", 0);
						//case a-b
						if (hyphenLoc != string::npos) {
							split_string_vector_type bonds;
							split(bonds, currentToken, is_any_of("-"), token_compress_on);
							int lowerBond = atoi(bonds.at(0).c_str());
							int upperBond = atoi(bonds.at(1).c_str());
							for (int i = lowerBond; i <= upperBond; i++) {
								trajs.push_back(i);
							}
						} else {
							int bond = atoi(currentToken.c_str());
							trajs.push_back(bond);
						}
					}
					//the following code lines ensure that all vector elements are unique
					vector<int>::iterator it;
					it = unique(trajs.begin(), trajs.end()); // 10 20 30 20 10 30 20 20 10
					trajs.resize(it - trajs.begin()); // 10 20 30 20 10
					settings -> setTrajs(trajs);
				} else if (strncmp(keyC, "supertraj_modes", 15) == 0) {
					vector<int> supertrajs;
					split_string_vector_type splitted;
					split(splitted, value, is_any_of(separator), token_compress_on); // #2: Search for tokens
					if(strlen(value.c_str()) == 0){
						continue;
					}
					for (uint i = 0; i < splitted.size(); i++) {
						string currentToken = splitted.at(i);
						string::size_type hyphenLoc = currentToken.find("-", 0);
						//case a-b
						if (hyphenLoc != string::npos) {
							split_string_vector_type bonds;
							split(bonds, currentToken, is_any_of("-"), token_compress_on);
							int lowerBond = atoi(bonds.at(0).c_str());
							int upperBond = atoi(bonds.at(1).c_str());
							for (int i = lowerBond; i <= upperBond; i++) {
								supertrajs.push_back(i);
							}
						} else {
							int bond = atoi(currentToken.c_str());
							supertrajs.push_back(bond);
						}
					}
					//the following code lines ensure that all vector elements are unique
					vector<int>::iterator it;
					it = unique(supertrajs.begin(), supertrajs.end()); // 10 20 30 20 10 30 20 20 10
					supertrajs.resize(it - supertrajs.begin()); // 10 20 30 20 10
					sort(supertrajs.begin(),supertrajs.end());
					settings -> setSuperTrajs(supertrajs);

				} else {
					delete[] writable;
					continue;
				}
			} else {
				delete[] writable;
				continue;
			}
			delete[] writable;
		}
		settings -> updateSettingsMap(setMap);
	} else {
		perror("Unable to open settings file. Please check the path you gave the program as first argument.");
		exit(1);
	}
	return true;
}

vector<vector<double> > FileHandler::readPQRM(const char* pqrmPath, const char* protName, BBMap &bbmap) {
	//each contained vector represents one coordinate axis
	vector<vector<double> > xyzCoordinates;
	vector<int> bbNumbers;
	list<string> temp;
	string line;

	char* fullPath = new char[strlen(pqrmPath)+strlen(protName)+strlen(".pqrm") + 1];
	fullPath = strcpy(fullPath,pqrmPath);
	fullPath = strcat(fullPath, protName);
	fullPath = strcat(fullPath,".pqrm");

	ifstream pqrmFile(fullPath);
	if (pqrmFile.is_open()) {
		while (pqrmFile.good()) {
			getline(pqrmFile, line);
			char *writable = new char[line.size() + 1];
			std::copy(line.begin(), line.end(), writable);
			writable[line.size()] = '\0';
			if (strncmp(writable, "ATOM  ", 6) == 0 or strncmp(writable, "HETATM  ", 8) == 0) {
				temp.push_back(line);
			} else {
				delete[] writable;
				continue;
			}
			delete[] writable;
		}

		int i = 0;
		int size = temp.size();
		//pre-init with N times 0.0
		vector<double> x_vector(size, 0.0);
		vector<double> y_vector(size, 0.0);
		vector<double> z_vector(size, 0.0);
		char *atom_type = new char[5];

		while (!temp.empty()) {
			string currentLineString = temp.front();
			char *writable = new char[currentLineString.size() + 1];
			std::copy(currentLineString.begin(), currentLineString.end(), writable);
			writable[currentLineString.size()] = '\0';
			sscanf(writable, "%*s %*s %s %*s %*d %lf %lf %lf %*s %*s %*lf\n", atom_type,&(x_vector.at(i)), &(y_vector.at(i)), &(z_vector.at(i)));
			if(strncmp(atom_type,"BB",2) == 0){
				bbNumbers.push_back(i);
			}
			temp.pop_front();
			i++;
			delete[] writable;
		}
		xyzCoordinates.push_back(x_vector);
		xyzCoordinates.push_back(y_vector);
		xyzCoordinates.push_back(z_vector);
		bbmap[protName] = bbNumbers;
		pqrmFile.close();
		delete[] fullPath;
		delete[] atom_type;
	} else{
		delete[] fullPath;
		perror("Unable to open pqrm file. Check file path to setup file and the configured protein names.");
		exit(1);
	}
	return xyzCoordinates;
}

vector<vector<double> > FileHandler::readEigenvaluesAndVecs(const char* path, const char* protName, const char* model, SettingsMap &setMap, MassMap &allMasses) {

	//create file path
	const char* fileName = "_eigValsAndVecs";
	const char* fileNameExtension = ".out";
	char* fullFileName = new char[strlen(path) + strlen(protName) + strlen(model) + strlen(fileName) + strlen(fileNameExtension) + 1];
	strcpy(fullFileName, path);
	strcat(fullFileName, protName);
	strcat(fullFileName, fileName);
	strcat(fullFileName, model);
	strcat(fullFileName, fileNameExtension);

	string line;
	string delim = " ";
	string separator = ",";
	int start = 0;
	if (strncmp(model, "ANM", 3) == 0) {
		start = 6;
	}

	vector<vector<double> > complete;

	ifstream eigValFile(fullFileName);
	if (eigValFile.is_open()) {
		int counter = 0;
		vector<double> currentEVV;
		double smallestEV = 0.0;
		while (eigValFile.good()) {
			getline(eigValFile, line);
			double currEV = 0.0;
			int brackPos = line.find("(");
			string massWHKey = "massWeightedHessian: ";
			int massWeightedHessianPos = line.find(massWHKey);
			string atomMassesKey = "Atom masses: ";
			int atomMassesPos = line.find(atomMassesKey);
			if (strncmp(line.c_str(), "EVal", 4) == 0) {
				if (counter >= start) {
					int pos = line.rfind(delim);
					string value = line.substr(pos + 1);
					currEV = atof(value.c_str());
					currentEVV.push_back(currEV);
					if(currEV < smallestEV or smallestEV == 0.0){
						smallestEV = currEV;
					}
				} else {
					counter++;
					continue;
				}
			} else if (brackPos != -1 and currentEVV.size() > 0) {
				int size = line.rfind(")");
				string values = line.substr(brackPos + 1, size - 2);
				split_string_vector_type splitted;
				split(splitted, values, is_any_of(separator), token_compress_on);
				for (uint i = 0; i < splitted.size(); i++) {
					string current = splitted.at(i);
					currentEVV.push_back(atof(current.c_str()));
				}
				complete.push_back(currentEVV);
				currentEVV.erase(currentEVV.begin(), currentEVV.end());
			} else if (massWeightedHessianPos != -1 and strncmp(model, "ANM", 3) == 0) {
				string value = line.substr(massWeightedHessianPos + strlen(massWHKey.c_str()), 1);
				setMap["massWeightedHessian"] = atof(value.c_str());
			} else if (atomMassesPos != -1 and strncmp(model, "ANM", 3) == 0) {
				string values = line.substr(atomMassesPos + strlen(atomMassesKey.c_str()));
				vector<double> masses;
				split_string_vector_type splitted;
				split(splitted, values, is_any_of(separator), token_compress_on);
				for (uint i = 0; i < splitted.size(); i++) {
					string cleanedValue = splitted.at(i);
					remove(cleanedValue.begin(), cleanedValue.end(), ' ');
					masses.push_back(atof(cleanedValue.c_str()));
				}
				allMasses[protName] = masses;
			}
		}
		if(setMap["supertraj_tstep"] == 0.0){
			setMap["supertraj_tstep"] = setMap["traj_tstep"]/sqrt(smallestEV);
		}
		delete[] fullFileName;
	} else {
		delete[] fullFileName;
		perror("Unable to open eigenvalue and -vector file. Please check the paths.");
		exit(1);
	}
	return complete;
}

FILE* FileHandler::createAndOpenDCDFile(int modeIndex, const char* protName, const char* path) {

	stringstream fileName;
	if(modeIndex != 0){
		fileName << path << protName << "_traj" << modeIndex << ".dcd";
	}else{
		fileName << path << protName << "_superTraj.dcd";
	}
	FILE* dcdFile = fopen(fileName.str().c_str(), "w");

	return dcdFile;
}

FILE* FileHandler::createAndOpenPDBFile(int modeIndex, const char* protName, const char* path) {

	stringstream fileName;
	if(modeIndex != 0){
		fileName << path << protName << "_traj" << modeIndex << ".pdb";
	}else{
		fileName << path << protName << "_superTraj.pdb";
	}
	FILE* pdbFile = fopen(fileName.str().c_str(), "w");
	return pdbFile;
}

FILE* FileHandler::createAndOpenDistFile(int modeIndex, const char* protName, const char* path) {

	stringstream fileName;
	if(modeIndex != 0){
		fileName << path << protName << "_traj" << modeIndex << ".dist";
	}else{
		fileName << path << protName << "_superTraj.dist";
	}
	FILE* distFile = fopen(fileName.str().c_str(), "w");
	return distFile;
}

void FileHandler::closeFile(FILE* file) {
	if (fclose(file) != 0) {
		perror("WARNING: File not closable!!");
	}
}

void FileHandler::writeDCDHeader(FILE* dcdFile, int natom, int trajSteps) {
	TrajectHeader header;
	this -> generateCHARMmHeader(&header, 1, natom, 1, 1, trajSteps);
	this -> writeCHARMmHeader(&header, dcdFile);
}

void FileHandler::writeDCD(FILE* dcdFile, float** coords, int natom) {

	int length = (int) sizeof(float) * natom;
	if (dcdFile != NULL) {
		//save x coords
		this -> saveWriteBinary(&length, sizeof(int), 1, dcdFile);
		this -> saveWriteBinary((coords[0]), sizeof(float), natom, dcdFile);
		this -> saveWriteBinary(&length, sizeof(int), 1, dcdFile);

		//save y coords
		this -> saveWriteBinary(&length, sizeof(int), 1, dcdFile);
		this -> saveWriteBinary((coords[1]), sizeof(float), natom, dcdFile);
		this -> saveWriteBinary(&length, sizeof(int), 1, dcdFile);

		//save z coords
		this -> saveWriteBinary(&length, sizeof(int), 1, dcdFile);
		this -> saveWriteBinary((coords[2]), sizeof(float), natom, dcdFile);
		this -> saveWriteBinary(&length, sizeof(int), 1, dcdFile);
	}
}

void FileHandler::writePseudoPDB(FILE* pdbFile, float** coords, int natom) {
	for (int i = 0; i < natom; i++) {
		fprintf(pdbFile, "ATOM  %5d    %8.3f%8.3f%8.3f\n", i, coords[0][i], coords[1][i], coords[2][i]);
	}
	fprintf(pdbFile, "TER\n");
	fprintf(pdbFile, "END\n");
}

void FileHandler::writeDists(FILE* distFile, vector< pair<double,float> > dists) {
	for (uint i = 0; i < dists.size(); i++) {
		pair<double,float> cur = dists.at(i);
		fprintf(distFile, "%f: %e\n", cur.first, cur.second);
	}
}

void FileHandler::saveWriteBinary(const void * ptr, int size, uint number, FILE* output) {

	uint checkBytes = 0;
	checkBytes = fwrite(ptr, size, number, output);
	if (number != checkBytes) {
		perror("Can not write a binary file");
	}
}

void FileHandler::generateCHARMmHeader(TrajectHeader *header, long seed, int natom, int n_start, int interv, int n_steps) {

#define TITLE "* This trajectory is from M. Ullmann!!  This trajectory is from M. Ullmann!!----"

	strcpy((*header).type, "CORD");

	(*header).info[N_SET] = n_steps / interv;
	(*header).info[N_START] = n_start;
	(*header).info[INTERV] = interv;
	(*header).info[N_STEPS] = n_steps;
	(*header).info[4] = 0;
	(*header).info[5] = 0;
	(*header).info[6] = 0;
	(*header).info[N_DOF] = 3 * natom;
	(*header).info[N_FIX] = 0;
	(*header).info[SEED] = (int) seed; /* CHANGE - should not be converted!! */
	(*header).info[PRES] = 0;
	(*header).info[11] = 0;
	(*header).info[12] = 0;
	(*header).info[13] = 0;
	(*header).info[14] = 0;
	(*header).info[15] = 0;
	(*header).info[16] = 0;
	(*header).info[17] = 0;
	(*header).info[18] = 0;
	(*header).info[CHARMM_VER] = 22;

	(*header).no_title = 2;
	(*header).title = charmatrix(1, 2, 0, 80);
	if (strlen(TITLE) != 80)
		perror("Title (CHARMM header) has a wrong length.");
	strcpy((*header).title[1], TITLE);
	strcpy((*header).title[2], TITLE);
	(*header).no_atoms = natom;

#undef TITLE
}

FILE* FileHandler::writeCHARMmHeader(TrajectHeader* header, FILE* dcdFile) {
	int length, i;

	/* read the first record */
	length = 20 * sizeof(int) + 4 * sizeof(char);
	this -> saveWriteBinary((&length), sizeof(int), 1, dcdFile);
	this -> saveWriteBinary((*header).type, sizeof(char), 4, dcdFile);
	this -> saveWriteBinary((*header).info, sizeof(int), 20, dcdFile);
	this -> saveWriteBinary((&length), sizeof(int), 1, dcdFile);

	/* read the second record */
	length = (int) sizeof(int) + 80 * (*header).no_title * (int) sizeof(char);
	this -> saveWriteBinary((&length), sizeof(int), 1, dcdFile);
	this -> saveWriteBinary(&((*header).no_title), sizeof(int), 1, dcdFile);
	for (i = 1; i <= (*header).no_title; i++) {
		this -> saveWriteBinary((*header).title[i], sizeof(char), 80, dcdFile);
	}
	this -> saveWriteBinary((&length), sizeof(int), 1, dcdFile);

	/* read the third record */
	length = sizeof(int);
	this -> saveWriteBinary(&length, sizeof(int), 1, dcdFile);
	this -> saveWriteBinary(&((*header).no_atoms), sizeof(int), 1, dcdFile);
	this -> saveWriteBinary(&length, sizeof(int), 1, dcdFile);

	//only for checking correctness of charmm header
	//FILE* headerFile = fopen("charmmHeader.out", "w");
	//this -> fprint_header_info(headerFile, header);

	return dcdFile;
}

void FileHandler::fprint_header_info(FILE *fpout, TrajectHeader *header) {
	int i;

	fprintf(fpout, "Kind of data:        %s\n", (*header).type);

	fprintf(fpout, "No of data sets:     %d\n", (*header).info[N_SET]);
	fprintf(fpout, "No of steps before:  %d\n", (*header).info[N_START]);
	fprintf(fpout, "Interval:            %d\n", (*header).info[INTERV]);
	fprintf(fpout, "Steps:		    %d\n", (*header).info[N_STEPS]);
	fprintf(fpout, "Don't know:	    %d\n", (*header).info[4]);
	fprintf(fpout, "Don't know:	    %d\n", (*header).info[5]);
	fprintf(fpout, "Don't know:	    %d\n", (*header).info[6]);
	fprintf(fpout, "Degress of Freedom:  %d\n", (*header).info[N_DOF]);
	fprintf(fpout, "Fixed Atoms:	    %d\n", (*header).info[N_FIX]);
	fprintf(fpout, "Seed:	  	    %d\n", (*header).info[SEED]);
	fprintf(fpout, "Pressure flag:	    %d\n", (*header).info[PRES]);
	fprintf(fpout, "Don't know:	    %d\n", (*header).info[11]);
	fprintf(fpout, "Don't know:	    %d\n", (*header).info[12]);
	fprintf(fpout, "Don't know:	    %d\n", (*header).info[13]);
	fprintf(fpout, "Don't know:	    %d\n", (*header).info[14]);
	fprintf(fpout, "Don't know:	    %d\n", (*header).info[15]);
	fprintf(fpout, "Don't know:	    %d\n", (*header).info[16]);
	fprintf(fpout, "Don't know:	    %d\n", (*header).info[17]);
	fprintf(fpout, "Don't know:	    %d\n", (*header).info[18]);
	fprintf(fpout, "Version:	  	    %d\n", (*header).info[CHARMM_VER]);

	fprintf(fpout, "number of title lines:	  %d\n", (*header).no_title);
	for (i = 1; i <= (*header).no_title; i++) {
		fprintf(fpout, "%s\n", (*header).title[i]);
	}

	fprintf(fpout, "Number of Atoms:   %d\n", (*header).no_atoms);
	fprintf(fpout, "alive /print header/\n");
}
