/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @date
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The ConfComparator class provides some functionalities to compare different conformations of the same protein. \
 * In order to do this the comparator holds reference coordinates of the target structure and time dependent coordinates basing on a vibration function.
 */

#ifndef CONFCOMPARATOR_H_
#define CONFCOMPARATOR_H_

#include "SetupConstants.h"
#include "FileHandler.h"

/*
 *
 */
class ConfComparator {
public:
	/*!
	 * This is the constructor for ConfComparator.
	 * @param settings These are the setup options you defined in your settings file.
	 * @param fileHandler This is an instance of an i/o manager called \link<FileHandler>FileHandler\endlink .
	 * @return nothing to return
	 */
	ConfComparator(FileHandler* fileHandler, SetupConstants* settings);

	/*!
	 * This is the standard destructor.
	 * @return nothing to return
	 */
	virtual ~ConfComparator();

	/*!
	 * This is not a normal setter method. It splits the two dimensional vector in three one dimensional vectors - one vector for each axis of coordinates.
	 * Afterwards it calculates a distance matrix for each pair of atoms.
	 * @param refCoords This vector contains three vectors. Each represents one axis of coordinates.
	 * @param refName This is the name of the reference protein.
	 */
	virtual void setRefCoords(vector<vector<double> > refCoords, const char* refName);
	virtual void setTimeStepCoords(float** timeStepCoords);

	/*!
	 * This method calculates the difference matrix of the current time dependent coordinates for each pair of atoms. Then it calculates the difference between this difference and the difference matrix of the reference protein.
	 * A variable holds the sum of differences between both matrices (added entry by entry). Each time step the sum of differences is added to a vector or the next higher value is replaced by this if the maximum number of smallest differences is reached.
	 * @param time This is the time step for which the current conformation of the protein is to be compared to the reference structure.
	 * @param protName This is the name of the protein which changes its conformation.
	 * @param refName This is the name of the reference structure.
	 * @return The sum, element by element, of differences between both difference matrices.
	 */
	virtual float calculateDiff(double time, const char* protName, const char* refName);

	/*!
	 * Returns the vector which contains a pair of a time step and a sum of differences at this time step. This vector is sorted starting with the smallest sum of differences.
	 * @return the vector with time step and sum of differences
	 */
	virtual vector<pair<double, float> > getMinimalDiff();

private:
	FileHandler* fileHandler;
	SetupConstants* settings;
	float** refDiffMatrix;
	float** timeStepCoords;
	vector<pair<double, float> > minimalDiff;
	uint coordinateColumnSize;
};

#endif /* CONFCOMPARATOR_H_ */
