/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @date
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The AbstractTrajectoryGenerator class is an abstract class which declares which functions a trajectory generator must have.
 * Child classes of this abstract class have to be able to calculate trajectories with the help of eigenvalues and -vectors. Furthermore they have to have a printing routine or have to be able to delegate this task.
 * DO NOT create an instance of this class. Extend this abstract class to implement a valid calculator.
 */

#ifndef ABSTRACTTRAJECTORYGENERATOR_H_
#define ABSTRACTTRAJECTORYGENERATOR_H_

#include <vector>
#include "SetupConstants.h"
#include "FileHandler.h"

/*
 *
 */
class AbstractTrajectoryGenerator {
public:
	/*!
	 * This is the constructor for the AbstractTrajectoryGenerator. DO NOT USE IT! If you use it, the compiler will throw an error.
	 * Instead using this constructor create an instance of a child class e.g. \link<TrajectoryGeneratorLapack>TrajectoryGeneratorLapack\endlink
	 * @param settings These are the setup options you defined in your settings file.
	 * @param fileHandler This is an instance of an i/o manager called \link<FileHandler>FileHandler\endlink .
	 * @return nothing to return
	 */
	AbstractTrajectoryGenerator(SetupConstants *settings, FileHandler* fileHandler);

	/*!
	 * This is the standard destructor of an object.
	 * @return nothing to return
	 */
	virtual ~AbstractTrajectoryGenerator();

	/*!
	 * This method calculates a single trajectory which means a vibration which bases upon ONE eigenvector/mode (specified via settings file). Furthermore the method prints this trajectory either as dcd file or as pdb file. For printing the method uses the fileHandler object.
	 * If configured the method also compares the protein conformation with a reference structure each time step.
	 * @param protName The name of the protein which is currently processed.
	 * @param startCoords The start coordinates for all relevant atoms which were read from the pqrm file.
	 * @param eigenvals The eigenvalues (3 times the number of atoms - 6)  of the hessian matrix. For more information please check the user manual.
	 * @param eigenvecs The eigenvectors (3 times the number of atoms - 6) of the hessian matrix. For more information please check the user manual.
	 * @param internalProtId The internal protein id which means the position of the protein name in the vector of protein names (see \link<main>main\endlink)
	 */
	virtual void calcAndPrintSingleTrajectories(const char* protName, vector<vector<double> > startCoords, vector<double> eigenvals, double* eigenvecs, int internalProtId) = 0;

	/*!
	 * This method calculates a super trajectory which means a vibration which bases upon all configured eigenvectors/modes (specified via settings file). Furthermore the method prints this super trajectory either as dcd file or as pdb file. For printing the method uses the fileHandler object.
	 * If configured the method also compares the protein conformation with a reference structure each time step.
	 * @param protName The name of the protein which is currently processed.
	 * @param startCoords The start coordinates for all relevant atoms which were read from the pqrm file.
	 * @param eigenvals The eigenvalues (3 times the number of atoms - 6)  of the hessian matrix. For more information please check the user manual.
	 * @param eigenvecs The eigenvectors (3 times the number of atoms - 6) of the hessian matrix. For more information please check the user manual.
	 * @param internalProtId The internal protein id which means the position of the protein name in the vector of protein names (see \link<main>main\endlink)
	 */
	virtual void calcAndPrintSuperTrajectories(const char* protName, vector<vector<double> > startCoords, vector<double> eigenvals, double* eigenvecs, int internalProtId) = 0;

	/*!
	 * The hessian matrix may be multiplied by M^-1/2 (2 times) which means that the hessian elements are weighted with the atom masses.
	 * In order to calculate the trajectories you have to recalculate the modified hessian matrix. For more information please have a look
	 * at the user manual.
	 * @param hessianMatrix This is the mass weighted hessian matrix.
	 * @param numberOfEVals This is 3 x the number of atoms - 6 (because the first 6 eigenvalues are zero).
	 * @return The hessian matrix with no orthogonal eigenvectors but instead of that correct mass weighted values. For more information please
	 * have a look at the user manual.
	 */
	virtual double* prepareHessianMatrix(double* modifiedHessianMatrix, int numberOfEVals, const char* protName) = 0;

	/*!
	 * The task of this method is to calculate a random value between -1 and +1 for all not manually configured phases.
	 * @param phaseMap The manually configured phases mapped to their modes.
	 * @param superTrajs Contains the indices of modes which are to be used for computing the super trajectory.
	 * @return
	 */
	virtual PhaseMap preparePhasesMap(PhaseMap phaseMap, vector<int> superTrajs) = 0;

private:
	virtual float** calculateNewCoords(vector<vector<double> > startCoords, double* displacementStep) = 0;
	virtual double* calculateSingleDisplacementStep(vector<double> linExpansions, double* eigenvecs, int trajIndex, double time) = 0;
	virtual double* calculateSuperDisplacementStep(vector<double> linExpansions, double* eigenvec, vector<double> eigenvals, int trajIndex, double time, PhaseMap phases) = 0;
	virtual vector<double> calculateLinearExpansions(vector<double> eigenvalues) = 0;

protected:
	SetupConstants* settings;
	FileHandler* fileHandler;
};

#endif /* ABSTRACTTRAJECTORYGENERATOR_H_ */
