/*---------------------------------------------------------
    This file is part of the program suite TEN
    (Tools for Elastic Networks)

    Copyright (C)

	Lars Ackermann
    G. Matthias Ullmann
    Bayreuth 2014

    www.bisb.uni-bayreuth.de

    This program is free software: you can redistribute
    it and/or modify it under the terms of the
    GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will
    be useful, but WITHOUT ANY WARRANTY; without even the
    implied warranty of MERCHANTABILITY or FITNESS FOR A
    PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the
    GNU Affero General Public License along with this
    program.  If not, see <http://www.gnu.org/licenses/>.
-----------------------------------------------------------*/

#include "main.h"
#include <iostream>
#include "stdio.h"
#include "Protein.h"
#include "FileHandler.h"
#include "SetupConstants.h"
#include "EigenValAndVecCalculatorLapack.h"
#include <stdlib.h>
#include <string.h>
#include <vector>
#include <time.h>
#include <errno.h>
using namespace std;

/*possible arguments in argv:
 * [0] - contains the command itself, which means the location of the executed program.
 * [1] - absolute path (including file name) to the setup file to use (there it is specified which proteins shall be processed [names ...], where they are [paths...]...)
 *
 *settings:
 *	protNames: names of proteins
 *	pqrmPaths: paths for pqrm files of the proteins for all protNames (only folder path; NO FILE NAME AT THE END)
 *
 * */
int main(int argc, char **argv) {

	clock_t init, final, allPrevious;
	init = clock();

	char* setupFilePath;
	if (argc < 2) {
		setupFilePath = "settings";
	} else {
		setupFilePath = argv[1];
	}

	SetupConstants* constants = new SetupConstants();
	FileHandler* fileHandler = new FileHandler();
	fileHandler->readSettings(constants, setupFilePath);
	SettingsMap settings = constants->getSettings();

	std::cout << "settings: " << std::endl;
	std::cout << "kcovA: " << settings["kcovA"] << std::endl;
	std::cout << "kncovA: " << settings["kncovA"] << std::endl;
	std::cout << "kcovG: " << settings["kcovG"] << std::endl;
	std::cout << "kncovG: " << settings["kncovG"] << std::endl;
	std::cout << "rcutG: " << settings["rcutG"] << std::endl;
	std::cout << "rcutA: " << settings["rcutA"] << std::endl;
	if(settings["interacMode"]>1)
		{
		cerr << "Error: Interaction mode must be 0 or 1." << endl;
		exit(1);
		}
	std::cout << "interacMode: " << settings["interacMode"] << "\n" << std::endl;

	vector<string> names = constants -> getNames();
	vector<string> pqrms = constants -> getPaths();

	for (uint i = 0; i < (names.size()); i++) {
		allPrevious = clock() - init;
		const char* currentName = names.at(i).c_str();
		string currentPath;
		if (i < pqrms.size()) {
			currentPath = pqrms.at(i);
			int pathLength = strlen(currentPath.c_str());
			char ending = currentPath.at(pathLength - 1);
			if (ending != '/') {
				currentPath.append("/");
			}
		} else {
			currentPath = "";
		}
		std::cout << "Processing protein " << currentName << ":" << std::endl;
		std::cout << "Reading pqrm from dir " << currentPath << "...";
		Protein* prot = fileHandler->readPQRM(currentPath.c_str(), currentName, constants);
		std::cout << "done" << std::endl;
		EigenValAndVecCalculator* evvc = new EigenValAndVecCalculatorLapack(prot, constants, fileHandler);
		std::cout << "Calculating and printing eigenvalues and -vectors (this may take some minutes)..." << std::flush;
		evvc -> calculateAndPrintEigenValsAndVecs();
		std::cout << "done" << std::endl;
		delete evvc;
		final = clock() - init - allPrevious;
		std::cout << "Time: "  << (double) final / ((double) CLOCKS_PER_SEC) << std::endl;
	}
	delete fileHandler;
	delete constants;
	std::cout << "All done... ";

	final = clock() - init;
	std::cout << "Total time: " << (double) final / ((double) CLOCKS_PER_SEC) << std::endl;

	return 0;
}
