/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @date
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The SetupConstants class is a data structure which holds all values of the user's settings file.
 * It consists of a map which contains all values of type double, a vector with the names of the proteins to be processed,
 * a vector with the paths to the pqrms and a vector with the network models which have to be used.
 */

#include <boost/unordered_map.hpp>
#include <vector>
#include <string.h>

#ifndef SETUPCONSTANTS_H_
#define SETUPCONSTANTS_H_

using namespace std;

typedef boost::unordered_map<std::string, double> SettingsMap;

class SetupConstants {

public:
	/*!
	 * This is the standard constructor. As the class doesn't have any parameters it is not necessary
	 * to define a specific constructor.
	 * @return nothing to return
	 */
	SetupConstants();

	/*!
	 * This is the standard destructor.
	 * @return nothing to return
	 */
	virtual ~SetupConstants();

	/*!
	 * This method overwrites the settings map with a new one. So make sure that you FIRST read out the old one
	 * using SetupConstants::readSettings, THEN manipulate values and FINALLY use this method.
	 * @param upToDateMap This is the new settings map which you want to overwrite the old with.
	 */
	virtual void updateSettingsMap(SettingsMap upToDateMap);

	/*!
	 * This method walks through the settings map and try to find the key inside.
	 * @param key This is the key to validate.
	 * @return True if the key is inside the map and false if it is not.
	 */
	virtual bool isKeyValid(string key);
	virtual SettingsMap getSettings();
	virtual vector<string> getNames();
	virtual vector<string> getPaths();
	virtual vector<string> getModels();
	virtual void setNames(vector<string> given);
	virtual void setPaths(vector<string> given);
	virtual void setModels(vector<string> given);

private:
	SettingsMap settings;
	vector<string> protNames;
	vector<string> pqrmPaths;
	vector<string> models;
};

#endif /* SETUPCONSTANTS_H_ */

