/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @date
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The Protein class is a data structure for holding data (in context) of the pqrm file during runtime.
 * "In context@ means that, e.g. the connect matrices for all network models are also hold here.
 */

#include <string.h>
#include <iostream>

#ifndef PROTEIN_H
#define PROTEIN_H

using namespace std;

class Protein
{
  public:
	/*!
	 * This is the constructor of the class Protein. The number of atoms is necessary to define the
	 * size of the arrays contained in this data structure.
	 * @param number_of_atoms This previously calculated value is the number of lines each containing
	 * information about one atom in the pqrm file.
	 * @return nothing to return
	 */
    Protein(const int number_of_atoms);
    
    /*!
     * This is the standard copy constructor.
     * @param prot The protein data structure to be copied.
     * @return nothing to return
     */
    Protein(const Protein &prot);
    
    /*!
     * This is the standard assignement operator of an object of type Protein.
     * @param right The Protein data structure to be assigned.
     * @return The assigned Protein data structure.
     */
    const Protein& operator=(const Protein &right);
    
    /*!
     * This is the standard destructor.
     * @return nothing to return
     */
    virtual ~Protein();
    
    virtual void addToXYZMatrix(double* xyzCurrent, int line);
    virtual void addToAtomTypeMatrix(char* atomTypeCurrent, int line);
    virtual void addToResidueMatrix(char* residueCurrent, int line);
    virtual void addToResNoMatrix(int resNumberCurrent, int line);
    virtual void addToMassMatrix(double massCurrent, int line);
    virtual void updateConnectGNMMatrix(double** connectCurrent);
    virtual void updateConnectANMMatrix(double** connectCurrent);
    virtual void addToBFactMatrix(double bfact, int line);
    virtual void setName(string protName);
    virtual void setPath(string protPath);
    virtual double* getFromXYZMatrix(int line);
    virtual char* getFromAtomTypeMatrix(int line);
    virtual char* getFromResidueMatrix(int line);
    virtual int getFromResNoMatrix(int line);
    virtual double getFromMassMatrix(int line);
    virtual double** getConnectGNMMatrix();
    virtual double** getConnectANMMatrix();
    virtual double getFromBFactMatrix(int line);
    virtual int getNumberOfAtoms();
    virtual string getName();
    virtual string getPath();
  
  private:
    double **xyz;         /*[N][3]*/
    char **atom_type;     /*[N][5]*/
    char **residue;       /*[N][4]*/
    int *res_no;          /* N */
    double *mass;         /* N */
    double **conectGNM;   /* N^2 */
    double **conectANM;	  /* N^2 */
    double *Bfact_exp;    /* N */
    int numberOfAtoms;
    string name;
    string path;
};

#endif // PROTEIN_H
