/*---------------------------------------------------------
    This file is part of the program suite TEN
    (Tools for Elastic Networks)

    Copyright (C)

	Lars Ackermann
    G. Matthias Ullmann
    Bayreuth 2014

    www.bisb.uni-bayreuth.de

    This program is free software: you can redistribute
    it and/or modify it under the terms of the
    GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will
    be useful, but WITHOUT ANY WARRANTY; without even the
    implied warranty of MERCHANTABILITY or FITNESS FOR A
    PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the
    GNU Affero General Public License along with this
    program.  If not, see <http://www.gnu.org/licenses/>.
-----------------------------------------------------------*/


#include "Protein.h"
#include <fstream>
#include <stdlib.h>

using namespace std;

Protein::Protein(const int number_of_atoms) {

	numberOfAtoms = number_of_atoms;
	// Dynamische Allokation auf dem Heap
	xyz = new double* [number_of_atoms];
	atom_type = new char* [number_of_atoms];
	residue = new char* [number_of_atoms];
	res_no = new int[number_of_atoms]; // N
	mass = new double[number_of_atoms]; // N
	conectGNM = new double*[number_of_atoms];
	conectANM = new double*[number_of_atoms];
	Bfact_exp = new double[number_of_atoms]; //N
	//allocate memory for 2D array contents
	for(int i=0; i < number_of_atoms; i++){
	   *(xyz+i) = new double[3]; /*[N][3]*/
	   *(atom_type+i) = new char[5]; /*[N][5]*/
	   *(residue+i) = new char[4]; /*[N][4]*/
	   *(conectGNM+i) = new double[number_of_atoms]; /*[N][N]*/
	   *(conectANM+i) = new double[number_of_atoms]; /*[N][N]*/
	}
}

Protein::Protein(const Protein &prot) {
	//memcpy is a highly-optimized method to copy arrays
	memcpy(xyz, prot.xyz, sizeof(prot.xyz));
	memcpy(atom_type, prot.atom_type, sizeof(prot.atom_type));
	memcpy(residue, prot.residue, sizeof(prot.residue));
	memcpy(res_no, prot.res_no, sizeof(prot.res_no));
	memcpy(mass, prot.mass, sizeof(prot.mass));
	memcpy(conectGNM, prot.conectGNM, sizeof(prot.conectGNM));
	memcpy(conectANM, prot.conectANM, sizeof(prot.conectANM));
	memcpy(Bfact_exp, prot.Bfact_exp, sizeof(prot.Bfact_exp));
}

const Protein& Protein::operator=(const Protein &right) {
	if(this != &right){
		int size = sizeof(right.res_no)/sizeof(int);
		//remember, that you can't just delete[] a multidimensional array - you need to delete it row by row, just as you create it.
		for(int i=0; i < size; i++){
			delete[] xyz[i];
			delete[] atom_type[i];
			delete[] residue[i];
			delete[] conectGNM[i];
			delete[] conectANM[i];
		}
		delete[] res_no;
		delete[] mass;
		delete[] Bfact_exp;
		delete[] xyz;
		delete[] atom_type;
		delete[] residue;
		delete[] conectGNM;
		delete[] conectANM;

		memcpy(xyz, right.xyz, sizeof(right.xyz));
		memcpy(atom_type, right.atom_type, sizeof(right.atom_type));
		memcpy(residue, right.residue, sizeof(right.residue));
		memcpy(res_no, right.res_no, sizeof(right.res_no));
		memcpy(mass, right.mass, sizeof(right.mass));
		memcpy(conectGNM, right.conectGNM, sizeof(right.conectGNM));
		memcpy(conectANM, right.conectANM, sizeof(right.conectANM));
		memcpy(Bfact_exp, right.Bfact_exp, sizeof(right.Bfact_exp));
	}
	return *this;
}

Protein::~Protein() {
	//remember, that you can't just delete[] a multidimensional array - you need to delete it row by row, just as you create it.
	for(int i=0; i < numberOfAtoms; i++){
		delete[] xyz[i];
		delete[] atom_type[i];
		delete[] residue[i];
		delete[] conectGNM[i];
		delete[] conectANM[i];
	}
	delete[] res_no;
	delete[] mass;
	delete[] Bfact_exp;
	delete[] xyz;
	delete[] atom_type;
	delete[] residue;
	delete[] conectGNM;
	delete[] conectANM;
}

void Protein::addToAtomTypeMatrix(char* atomTypeCurrent, int line) {
	memcpy(atom_type[line], atomTypeCurrent, 5);
}

void Protein::updateConnectGNMMatrix(double** connectCurrent) {
	conectGNM = connectCurrent;
}

void Protein::updateConnectANMMatrix(double** connectCurrent) {
	conectANM = connectCurrent;
}

void Protein::addToResidueMatrix(char* residueCurrent, int line) {
	memcpy(residue[line], residueCurrent, 4);
}

void Protein::addToXYZMatrix(double* xyzCurrent, int line) {
	memcpy(xyz[line],xyzCurrent,sizeof(xyzCurrent)*3);
}

void Protein::addToBFactMatrix(double bfact, int line) {
	Bfact_exp[line] = bfact;
}

void Protein::addToMassMatrix(double massCurrent, int line) {
	mass[line] = massCurrent;
}

void Protein::addToResNoMatrix(int resNumberCurrent, int line) {
	res_no[line] = resNumberCurrent;
}

void Protein::setName(string protName) {
	name = protName;
}

void Protein::setPath(string protPath){
	path = protPath;
}

double* Protein::getFromXYZMatrix(int line) {
	return xyz[line];
}
char* Protein::getFromAtomTypeMatrix(int line) {
	return atom_type[line];
}

char* Protein::getFromResidueMatrix(int line) {
	return residue[line];
}

int Protein::getFromResNoMatrix(int line) {
	return res_no[line];
}

double Protein::getFromMassMatrix(int line) {
	return mass[line];
}

double** Protein::getConnectGNMMatrix() {
	return conectGNM;
}

double** Protein::getConnectANMMatrix() {
	return conectANM;
}

double Protein::getFromBFactMatrix(int line) {
	return Bfact_exp[line];
}

int Protein::getNumberOfAtoms(){
	return numberOfAtoms;
}

string Protein::getName(){
	return name;
}

string Protein::getPath(){
	return path;
}
