/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @date
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The FileHandler class is responsible for input and output actions. It reads the
 * user's settings file, the pqrm file and writes the eigenvalues and -vectors to files.
 */

#ifndef FILEHANDLER_H
#define FILEHANDLER_H
#include "Protein.h"
#include "SetupConstants.h"

class FileHandler
{
  public:
    
	/*!
	 * This is the standard constructor for the FileHandler. A specific one is not necessary
	 * because the FileHandler doesn't have any parameters.
	 * @return nothing
	 */
    FileHandler();
	/*!
	 * This is the standard copy constructor of an object.
	 * @return nothing to return
	 */
    FileHandler(const FileHandler &right);
	/*!
	 * This is the standard destructor of an object.
	 * @return nothing to return
	 */
    ~FileHandler();
	/*!
	 * This is the standard assignment operator of an object of type FileHandler.
	 * @return the assigned FileHandler
	 */
    const FileHandler& operator=(const FileHandler &right);
    
    /*!
     * This method reads the settings defined in the file named "settings". Different settings for one
     * run is not possible, yet.
     * @param settings This refers to the settings object which is used all over the program.
     * @param settingsPath This is the path where the settings file can be found. Can be an empty string and can be defined via command line argument on program execution.
     * @return True if all settings have been read.
     */
    virtual bool readSettings(SetupConstants* settings, char* settingsPath);

    /*!
     * This method reads the pqrm file for one protein and stores data in the protein data structure. It holds an initial connection matrix for each network model
     * which has to be processed. These connection matrices are build with the help of connect entries of the pqrm files.
     * @param pqrmPath This is the path were the pqrm file of the protein can be found. Can be an empty string and can be defined in the settings file.
     * @param protName This is the name of the protein which has to be processed. This name must NOT be an empty string. It the pqrm file name must be the same.
     * @param settings This refers to the settings object which is used all over the program.
     * @return The data structure of type Protein with all contents of the pqrm in context.
     */
    virtual Protein* readPQRM(const char* pqrmPath, const char* protName, SetupConstants* settings);

    /*!
     * This method writes all eigenvalues and eigenVecs to a file with the name of the protein concatenated with the type of model. It also writes
     * the settings used for computation.
     * @param eigenVals This array contains all eigenvalues of the network model's interaction matrix.
     * @param eigenVecs This array contains all eigenvectors of the network model's interaction matrix.
     * @param numberOfAtoms This is the number of atoms which have been focused. It is used to decide how many eigenvalues and -vectors have to be written.
     * @param model This defines the name of the model the eigenvalues and -vectors belong to.
     * @param protPath This defines the path to the output folder.
     * @param protName This defines the name of the protein which has been processed.
     * @param settings This refers to the settings object which is used all over the program.
     * @return True if the whole content was written to the output file.
     */
    virtual bool writeEigenvaluesAndVectors(double* eigenVals, double* eigenVecs, int numberOfAtoms, const char* model, const char* protName, const char* protPath, SetupConstants* settings, Protein* prot);

    /*!
     * This method uses the zip utils to zip the eigenvalue and -vector files.
     * @param fullFileName The file name of the eigenvalue and -vector file including the path.
     */
    void compress(const char* fullFileName);
};

#endif // FILEHANDLER_H
