/*---------------------------------------------------------
    This file is part of the program suite TEN
    (Tools for Elastic Networks)

    Copyright (C)
	
	Lars Ackermann
    G. Matthias Ullmann
    Bayreuth 2014

    www.bisb.uni-bayreuth.de

    This program is free software: you can redistribute
    it and/or modify it under the terms of the
    GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will
    be useful, but WITHOUT ANY WARRANTY; without even the
    implied warranty of MERCHANTABILITY or FITNESS FOR A
    PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the
    GNU Affero General Public License along with this
    program.  If not, see <http://www.gnu.org/licenses/>.
-----------------------------------------------------------*/


#include "FileHandler.h"
#include <string.h>
#include <fstream>
#include <iostream>
#include "Protein.h"
#include "SetupConstants.h"
#include <list>
#include <algorithm>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/split.hpp>
#include <vector>
#include "stdio.h"

using namespace std;
using namespace boost::algorithm;

typedef vector<string> split_vector_type;

FileHandler::FileHandler() {

}

FileHandler::FileHandler(const FileHandler &right) {
}

FileHandler::~FileHandler() {
}

const FileHandler& FileHandler::operator=(const FileHandler &right) {
	return *this;
}

Protein* FileHandler::readPQRM(const char* pqrmPath, const char* protName, SetupConstants* settings) {
	Protein* prot;
	list<string> temp;
	list<string> conectTemp;
	string line;
	SettingsMap settingsMap = settings -> getSettings();
	vector<string> models = settings -> getModels();

	char* fullPath = new char[strlen(pqrmPath) + strlen(protName) + strlen(".pqrm") + 1];
	fullPath = strcpy(fullPath, pqrmPath);
	fullPath = strcat(fullPath, protName);
	fullPath = strcat(fullPath, ".pqrm");
	ifstream pqrmFile(fullPath);
	if (pqrmFile.is_open()) {
		while (pqrmFile.good()) {
			getline(pqrmFile, line);
			char writable[line.size() + 1];
			std::copy(line.begin(), line.end(), writable);
			//writable[line.size()] = '\0';
			if (strncmp(writable, "ATOM  ", 6) == 0 or strncmp(writable, "HETATM  ", 8) == 0) {
				temp.push_back(line);
			} else if (strncmp(writable, "CONECT  ", 8) == 0) {
				conectTemp.push_back(line);
			} else {
				continue;
			}
		}

		prot = new Protein(temp.size());
		prot -> setName(protName);
		prot -> setPath(pqrmPath);
		int i = 0;
		double *xyz = new double[3];
		char *atom_type = new char[5];
		char *residue = new char[4];
		int res_no;
		double mass;
		while (!temp.empty()) {
			string currentLineString = temp.front();
			char *writable = new char[currentLineString.size() + 1];
			std::copy(currentLineString.begin(), currentLineString.end(), writable);
			writable[currentLineString.size()] = '\0';
			sscanf(writable, "%*s %*s %s %s %d %lf %lf %lf %*s %*s %lf\n", atom_type, residue, &res_no, &xyz[0], &xyz[1], &xyz[2], &mass);
			prot->addToResNoMatrix(res_no, i);
			prot->addToAtomTypeMatrix(atom_type, i);
			prot->addToResidueMatrix(residue, i);
			prot->addToXYZMatrix(xyz, i);
			prot->addToMassMatrix(mass, i);
			temp.pop_front();
			i++;
			delete[] writable;
		}
		while (!conectTemp.empty()) {
			string currentLineString = conectTemp.front();
			char *writable = new char[currentLineString.size() + 1];
			std::copy(currentLineString.begin(), currentLineString.end(), writable);
			writable[currentLineString.size()] = '\0';
			char* tok = strtok(writable, " ");
			vector<string> parts;
			while (tok != NULL) {
				if (strncmp(tok, "CONECT", 6) == 0) {
					tok = strtok(NULL, " ");
					continue;
				}
				parts.push_back(tok);
				tok = strtok(NULL, " ");
			}
			int whichAtom = atoi(parts.at(0).c_str()) - 1;

			for (uint i = 1; i < parts.size(); i++) {
				//the -1 is because the index of kirchhoffMatrix begins at zero
				int connectedTo = atoi(parts.at(i).c_str()) - 1;
				if (find(models.begin(), models.end(), "GNM") != models.end()) {
					double** kirchhoffMatrix = prot -> getConnectGNMMatrix();
					kirchhoffMatrix[whichAtom][connectedTo] = -settingsMap["kcovG"];
				}
				if (find(models.begin(), models.end(), "ANM") != models.end()) {
					double** kirchhoffMatrix = prot -> getConnectANMMatrix();
					kirchhoffMatrix[whichAtom][connectedTo] = -settingsMap["kcovA"];
				}
			}
			conectTemp.pop_front();
			delete[] writable;
		}
		delete[] xyz;
		delete[] atom_type;
		delete[] residue;
		pqrmFile.close();
		delete[] fullPath;
	} else {
		perror("Unable to open pqrm file. Check file path to setup file and the configured protein names.");
		delete[] fullPath;
		exit(1);
	}
	return prot;
}

bool FileHandler::readSettings(SetupConstants* settings, char* settingsPath) {
	SettingsMap setMap = settings->getSettings();
	string delim = " ";
	string separator = ",";
	string line;
	ifstream settingsFile(settingsPath);
	if (settingsFile.is_open()) {
		while (settingsFile.good()) {
			getline(settingsFile, line);
			char writable[line.size() + 1];
			std::copy(line.begin(), line.end(), writable);
			writable[line.size()] = '\0';
			string::size_type loc = line.find(delim, 0);
			//If the initial two characters are '//' then continue because '//' is for comments
			if (loc != string::npos && strncmp(line.c_str(), "//", 2) != 0) {
				string key = line.substr(0, loc);
				remove(key.begin(), key.end(), ' ');
				string value = line.substr(loc + 1);
				if (settings->isKeyValid(key) and strlen(value.c_str()) != 0) {
					remove(value.begin(), value.end(), ' ');
					double doubleValue = atof(value.c_str());
					setMap[key] = doubleValue;
				} else if (strncmp(key.c_str(), "names", 5) == 0) {
					// split string for each separator
					split_vector_type splitted; // #2: Search for tokens
					split(splitted, value, is_any_of(separator), token_compress_on);
					settings -> setNames(splitted);
					//if separator found split string for each separator
				} else if (strncmp(key.c_str(), "protPaths", 9) == 0) {
					split_vector_type splitted; // #2: Search for tokens
					split(splitted, value, is_any_of(separator), token_compress_on);
					settings -> setPaths(splitted);
				} else if (strncmp(key.c_str(), "models", 6) == 0) {
					split_vector_type splitted; // #2: Search for tokens
					split(splitted, value, is_any_of(separator), token_compress_on);
					//The eigenvalues and -vectors for GNM are calculated if no model was given in the settings!
					if (splitted.empty()) {
						splitted.push_back("GNM");
					}
					settings -> setModels(splitted);
				} else {
					continue;
				}
			} else {
				continue;
			}
		}
		settings -> updateSettingsMap(setMap);
	} else {
		perror("Unable to open settings file. Please check the path you gave the program as first argument.");
		exit(1);
	}
	return true;
}

bool FileHandler::writeEigenvaluesAndVectors(double* eigenVals, double* eigenVecs, int numberOfAtoms, const char* model, const char* protName, const char* protPath, SetupConstants* settings,
		Protein* prot) {

	SettingsMap setMap = settings -> getSettings();

	//create file path
	const char* fileName = "_eigValsAndVecs";
	const char* fileNameExtension = ".out";
	char* fullFileName = new char[strlen(protName) + strlen(protPath) + strlen(model) + strlen(fileName) + strlen(fileNameExtension) + 1];
	strcpy(fullFileName, protPath);
	strcat(fullFileName, protName);
	strcat(fullFileName, fileName);
	strcat(fullFileName, model);
	strcat(fullFileName, fileNameExtension);

	ifstream eigValVecOutputFileToDelete(fullFileName);
	if (eigValVecOutputFileToDelete) {
		eigValVecOutputFileToDelete.close();
		remove(fullFileName);
	}
	FILE* eigValVecOutputFile;
	/* open the file*/
	if ((eigValVecOutputFile = fopen(fullFileName, "a+")) == NULL) {
		perror("Could not create and/or open the file with the eigenvalues and -vectors. Make sure you have closed it and you have the rights for writing within their directory.");
		exit(1);
	} else {
		if (strncmp(model, "GNM", 3) == 0) {
			fprintf(eigValVecOutputFile, "Used Settings: [kcovG: %.2f, kncovG: %.2f, rcutG: %.2f, interacMode: %.0f]\n\n\n", setMap["kcovG"], setMap["kncovG"], setMap["rcutG"], setMap["interacMode"]);
			for (int i = 0; i < numberOfAtoms; i++) {
				double eigenVal = eigenVals[i];
				fprintf(eigValVecOutputFile, "EVal%d: %g\nEVec%d: ", i + 1, eigenVal, i + 1);
				for (int j = 0; j < numberOfAtoms; j++) {
					double eigenVecValueAtIndex = eigenVecs[i * numberOfAtoms + j];
					if (j == 0) {
						fprintf(eigValVecOutputFile, "(%g,", eigenVecValueAtIndex);
					} else if (j < numberOfAtoms - 1) {
						fprintf(eigValVecOutputFile, "%g,", eigenVecValueAtIndex);
					} else {
						fprintf(eigValVecOutputFile, "%g)\n\n", eigenVecValueAtIndex);
					}
				}
			}
		} else if (strncmp(model, "ANM", 3) == 0) {
			fprintf(eigValVecOutputFile, "Used Settings: [kcovA: %.2f, kncovA: %.2f, rcutA: %.2f, interacMode: %.0f, massWeightedHessian: %.0f]\n\n\n", setMap["kcovA"], setMap["kncovA"],
					setMap["rcutA"], setMap["interacMode"], setMap["massWeightedHessian"]);
			fprintf(eigValVecOutputFile, "Atom masses: ");
			for (int i = 0; i < numberOfAtoms; i++) {
				if (i == 0) {
					fprintf(eigValVecOutputFile, "%.3f", prot -> getFromMassMatrix(i));
				} else {
					fprintf(eigValVecOutputFile, ", %.3f", prot -> getFromMassMatrix(i));
				}
			}
			fprintf(eigValVecOutputFile, "\n\n");
			for (int i = 0; i < numberOfAtoms * 3; i++) {
				double eigenVal = eigenVals[i];
				fprintf(eigValVecOutputFile, "EVal%d: %g\nEVec%d: ", i + 1, eigenVal, i + 1);
				for (int j = 0; j < numberOfAtoms * 3; j++) {
					double eigenVecValueAtIndex = eigenVecs[i * numberOfAtoms * 3 + j];
					if (j == 0) {
						fprintf(eigValVecOutputFile, "(%g,", eigenVecValueAtIndex);
					} else if (j < numberOfAtoms * 3 - 1) {
						fprintf(eigValVecOutputFile, "%g,", eigenVecValueAtIndex);
					} else {
						fprintf(eigValVecOutputFile, "%g)\n\n", eigenVecValueAtIndex);
					}
				}
			}
		}
		fclose(eigValVecOutputFile);
		delete[] fullFileName;
	}

	return true;
}

void FileHandler::compress(const char* fullFileName) {
}
