/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @date
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The EigenValAndVecCalculatorLapack class is an implementation of the abstract class EigenValAndVecCalculator.
 * It uses the library Lapack++ for the calculation. This in turn uses the original Fortran implementation of
 * Lapack and so Lapack++ is only a wrapper.
 */

#ifndef EIGENVALANDVECCALCULATORLAPACK_H_
#define EIGENVALANDVECCALCULATORLAPACK_H_

#include "EigenValAndVecCalculator.h"

class EigenValAndVecCalculatorLapack: public EigenValAndVecCalculator {
public:
	/*!
	 * This is the constructor you have to use if you need an instance of EigenValAndVecCalculator. It uses the
	 * super constructor of EigenValAndVecCalculator. For more information please have a look at the documentation
	 * of the super constructor.
	 * @param prot1
	 * @param settings1
	 * @param fileHandler
	 * @return
	 */
	EigenValAndVecCalculatorLapack(Protein* prot1, SetupConstants* settings1, FileHandler* fileHandler);

	/*!
	 * This is the standard destructor of an object.
	 * @return nothing to return
	 */
	virtual ~EigenValAndVecCalculatorLapack();

	/*!
	 * This method calculates the kirchhoff matrix for a given model. There are three possible ways to do it.
	 * Which mode is going to be processed is defined by the user's value of the option "interacMode" in the settings
	 * file.
	 * <ol>
	 * 	<li>interacMode = 0: The interaction between two atoms is defined by three discrete values. The major interaction
	 * constant is used for covalent bindings. The minor one is used for those atoms which are not bound directly but have a very small distance to each other
	 * (smaller than the cut-off value specified by the user). The last value is 0 and says that there is no interaction between two atoms.</li>
	 * 	<li>interacMode = 1: The interaction between two values is defined weighted. The covalent bindings are fixed but any other interaction is calculated
	 * depending on the distance between two atoms - even for those which have very long distances.</li>
	 * </ol>
	 * @param model
	 * @return
	 */
	virtual double** buildKirchhoffMatrix(const char* model);

	/*!
	 * This method calculates the hessian matrix from a given kirchhoff matrix. The hessian matrix is constructed by replacing each kirchhoff element with a 3x3 matrix.\
	 * For the exact formula please have a look at the user manual in the theory chapter.
	 * @param kirchhoffMatrix This is the previously calculated kirchhoff matrix for the anisotropic model.
	 * @return the hessian matrix
	 */
	virtual double** buildInteractionMatrixANM(double** kirchhoffMatrix);

	/*!
	 * This method calculates the distance between two given atoms. The coordinates used are specified in the pqrm file.
	 * The used metric is the Euclidean Distance.
	 * @param xyz_atom1 This array contains the x-y-z coordinates for the first atom.
	 * @param xyz_atom2 This array contains the x-y-z coordinates for the second atom.
	 * @return the distance between the two atoms.
	 */
	virtual double calculateDistance(double* xyz_atom1, double* xyz_atom2 );

	/*!
	 *	This method calculates the eigenvalues and -vectors using the Lapack routines. To be able to use Lapack it is
	 *	necessary that you transform the kirchhoff and accordingly the hessian matrix (both two dimensional arrays) into a
	 *	one-dimensional array that way that the array contains one column of the matrix after another. This restriction is given by the fortran
	 *	routines. The current implementation only wraps the Lapack++ data structure LaGenMatDouble around it and during the calculation the mentioned
	 *	one-dimensional array is edited. Finally the one-dimensional array contains the eigenvectors in one after another.
	 */
	virtual void calculateAndPrintEigenValsAndVecs();
};

#endif /* EIGENVALANDVECCALCULATORLAPACK_H_ */
