/*---------------------------------------------------------
    This file is part of the program suite TEN
    (Tools for Elastic Networks)

    Copyright (C)

	Lars Ackermann
    G. Matthias Ullmann
    Bayreuth 2014

    www.bisb.uni-bayreuth.de

    This program is free software: you can redistribute
    it and/or modify it under the terms of the
    GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will
    be useful, but WITHOUT ANY WARRANTY; without even the
    implied warranty of MERCHANTABILITY or FITNESS FOR A
    PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the
    GNU Affero General Public License along with this
    program.  If not, see <http://www.gnu.org/licenses/>.
-----------------------------------------------------------*/


#include "EigenValAndVecCalculatorLapack.h"
#include "EigenValAndVecCalculator.h"
#include "math.h"
#include "string.h"
#include <algorithm>
#include <vector>
//lapack++ library for LaGenMatDouble (matrix of doubles)
#include "lavd.h"
#include "gmd.h"
#include "laslv.h"

EigenValAndVecCalculatorLapack::EigenValAndVecCalculatorLapack(Protein* prot1, SetupConstants* settings1, FileHandler* fileHandler) :
	EigenValAndVecCalculator(prot1, settings1, fileHandler) {
}

EigenValAndVecCalculatorLapack::~EigenValAndVecCalculatorLapack() {
}

void EigenValAndVecCalculatorLapack::calculateAndPrintEigenValsAndVecs() {
	vector<string> models = settings -> getModels();
	int n_rows = prot -> getNumberOfAtoms();

	if (find(models.begin(), models.end(), "GNM") != models.end()) {
		double** kirchhoffMatrix = this -> buildKirchhoffMatrix("GNM");
		//FORTRAN needs a one-dimensional array which contains the values of one column after another.
		double* kirchhoffColByCol = new double[n_rows * n_rows];
		double* eigValsArray = new double[n_rows];
		for (int i = 0; i < n_rows; i++) {
			for (int j = 0; j < n_rows; j++) {
				kirchhoffColByCol[i * n_rows + j] = kirchhoffMatrix[i][j];
			}
		}
		LaGenMatDouble matr = LaGenMatDouble(kirchhoffColByCol, n_rows, n_rows, false);
		LaVectorDouble eigVals = LaVectorDouble(n_rows);
		//matr contains the orthonormal eigenvectors after following calculation
		//eigVals contains the eigenvalues after following calculation
		LaEigSolveSymmetricVecIP(matr, eigVals);
		for (int i = 0; i < n_rows; i++) {
			eigValsArray[i] = eigVals(i);
		}
		fileHandler -> writeEigenvaluesAndVectors(eigValsArray, kirchhoffColByCol, n_rows, "GNM", prot->getName().c_str(), (prot->getPath()).c_str(), settings, prot);
		delete[] kirchhoffColByCol;
		delete[] eigValsArray;
	}
	if (find(models.begin(), models.end(), "ANM") != models.end()) {
		double** kirchhoffMatrix = this -> buildKirchhoffMatrix("ANM");
		double** hessianMatrix = this -> buildInteractionMatrixANM(kirchhoffMatrix);

		//FORTRAN needs a one-dimensional array which contains the values of one column after another.
		double* hessianColByCol = new double[n_rows * 3 * n_rows * 3];
		double* eigValsArray = new double[n_rows * 3];
		for (int i = 0; i < n_rows * 3; i++) {
			for (int j = 0; j < n_rows * 3; j++) {
				hessianColByCol[i * n_rows * 3 + j] = hessianMatrix[i][j];
			}
		}
		LaGenMatDouble matr = LaGenMatDouble(hessianColByCol, n_rows * 3, n_rows * 3, false);
		LaVectorDouble eigVals = LaVectorDouble(n_rows * 3);

		//matr contains the orthonormal eigenvectors after following calculation
		//eigVals contains the eigenvalues after following calculation
		LaEigSolveSymmetricVecIP(matr, eigVals);
		for (int i = 0; i < n_rows * 3; i++) {
			eigValsArray[i] = eigVals(i);
			//std::cout << "eigval no. " << i <<  " " << eigVals(i) << std::endl;
		}
		fileHandler -> writeEigenvaluesAndVectors(eigValsArray, hessianColByCol, n_rows, "ANM", prot->getName().c_str(), (prot->getPath()).c_str(), settings, prot);
		delete[] eigValsArray;
		delete[] hessianColByCol;
		for (int i = 0; i < n_rows * 3; i++) {
			delete[] hessianMatrix[i];
		}
		delete[] hessianMatrix;
	}
}

double** EigenValAndVecCalculatorLapack::buildKirchhoffMatrix(const char* model) {

	SettingsMap settingsMap = settings -> getSettings();
	string bigInteracConst = "null";
	string smallInteracConst = "null";
	string cutOff = "null";
	double** connMatrix = NULL;
	if (strncmp("GNM", model, 3) == 0) {
		bigInteracConst = "kcovG";
		smallInteracConst = "kncovG";
		cutOff = "rcutG";
		connMatrix = prot -> getConnectGNMMatrix();

	} else if (strncmp(model, "ANM", 3) == 0) {
		bigInteracConst = "kcovA";
		smallInteracConst = "kncovA";
		cutOff = "rcutA";
		connMatrix = prot -> getConnectANMMatrix();
	}

	int numberOfAtoms = prot -> getNumberOfAtoms();
	if (settingsMap["interacMode"] == 0) {
		for (int i = 0; i < numberOfAtoms; i++) {
			//lineSum: for the diagonal elements
			double lineSum = 0.0;
			for (int j = 0; j < numberOfAtoms; j++) {
				double interacConstant = -connMatrix[i][j];
				if (interacConstant == settingsMap[bigInteracConst] and i != j) {
					lineSum = lineSum + connMatrix[i][j];
					continue;
				} else {
					double dist = this -> calculateDistance(prot -> getFromXYZMatrix(i), prot -> getFromXYZMatrix(j));
					if (dist <= settingsMap[cutOff] and i != j) {
						connMatrix[i][j] = -settingsMap[smallInteracConst];
						lineSum = lineSum + connMatrix[i][j];
					} else {
						connMatrix[i][j] = 0.0;
					}
				}
			}
			connMatrix[i][i] = -lineSum;
		}
	} 
//else if (settingsMap["interacMode"] == 1) {
//		for (int i = 0; i < numberOfAtoms; i++) {
			//lineSum: for the diagonal elements
//			double lineSum = 0.0;
//			for (int j = 0; j < numberOfAtoms; j++) {
//				double interacConstant = -connMatrix[i][j];
//				if (interacConstant == settingsMap[bigInteracConst] and i != j) {
//					lineSum = lineSum + connMatrix[i][j];
//					continue;
//				} else {
//					double dist = this -> calculateDistance(prot -> getFromXYZMatrix(i), prot -> getFromXYZMatrix(j));
//					if (dist <= settingsMap[cutOff] and i != j) {
//						double interactConst = settingsMap[bigInteracConst] / 2 * exp(-pow(dist, 2) / pow(settingsMap[cutOff], 2));
//						connMatrix[i][j] = -interactConst;
//						lineSum = lineSum + connMatrix[i][j];
//					} else {
//						connMatrix[i][j] = 0.0;
//					}
//				}
//			}
//			connMatrix[i][i] = -lineSum;
//		}
//	} 
	else if (settingsMap["interacMode"] == 1) {
		for (int i = 0; i < numberOfAtoms; i++) {
			//lineSum: for the diagonal elements
			double lineSum = 0.0;
			for (int j = 0; j < numberOfAtoms; j++) {
				double interacConstant = -connMatrix[i][j];
				if (interacConstant == settingsMap[bigInteracConst] and i != j) {
					lineSum = lineSum + connMatrix[i][j];
					continue;
				} else if (i == j) {
					connMatrix[i][j] = 0.0;
				} else {
					double dist = this -> calculateDistance(prot -> getFromXYZMatrix(i), prot -> getFromXYZMatrix(j));
					double interactConst = settingsMap[bigInteracConst] / 2 * exp(-pow(dist, 2) / pow(settingsMap[cutOff], 2));
					connMatrix[i][j] = -interactConst;
					lineSum = lineSum + connMatrix[i][j];
				}
			}
			connMatrix[i][i] = -lineSum;
		}
	}

	//prot -> updateConnectGNMMatrix(connMatrix);
	return connMatrix;
}

double** EigenValAndVecCalculatorLapack::buildInteractionMatrixANM(double** kirchhoffMatrix) {

	SettingsMap settingsMap = settings -> getSettings();

	int number_of_atoms = prot -> getNumberOfAtoms();
	double** hessianMatrix = new double*[number_of_atoms * 3];
	for (int i = 0; i < number_of_atoms * 3; i++) {
		double* line = new double[number_of_atoms * 3];
		std::fill_n(line, number_of_atoms * 3, 0);
		*(hessianMatrix + i) = line;
	}
	for (int i = 0; i < number_of_atoms; i++) {
		for (int j = 0; j < number_of_atoms; j++) {
			if (i == j) {
				//Because the diagonal elements of the Hessian matrix are calculated as the sum of the corresponding line, the calculation have to be done later.
				continue;
			} else {
				double* xyz_atom1 = prot -> getFromXYZMatrix(i);
				double* xyz_atom2 = prot -> getFromXYZMatrix(j);
				double dist = this -> calculateDistance(xyz_atom1, xyz_atom2);
				double kirchNormDist = kirchhoffMatrix[i][j] / pow(dist, 2);
				//The following two loops are necessary because the hessian matrix contains for each kirchhoff element a 3x3 matrix which is calculated in as follows.
				for (int k = 0; k < 3; k++) {
					for (int l = 0; l < 3; l++) {
						//fill in one element of the Hessian matrix.
						hessianMatrix[(i * 3) + k][(j * 3) + l] = kirchNormDist * ((xyz_atom1[k] - (xyz_atom2[k])) * (xyz_atom1[l] - (xyz_atom2[l])));
					}
				}
			}
		}
	}

	//now the diagonal elements
	for (int i = 0; i < number_of_atoms; i++) {
		for (int k = 0; k < 3; k++) {
			for (int l = 0; l < 3; l++) {
				//fill in one diagonal element of the Hessian matrix.
				double sum = 0.0;
				for (int j = 0; j < number_of_atoms; j++) {
					sum = sum + hessianMatrix[(i * 3) + k][(j * 3) + l];
				}
				hessianMatrix[(i * 3) + k][(i * 3) + l] = -sum;
			}
		}
	}

	/*	for (int i = 0; i < number_of_atoms * 3; i++) {
	 for (int j = 0; j < number_of_atoms * 3; j++) {
	 std::cout << hessianMatrix[i][j] << " ";
	 }
	 std::cout << std::endl;
	 }*/

	//now the mass is used to weight the hessian elements
	if (settingsMap["massWeightedHessian"] == 1) {
		for (int i = 0; i < number_of_atoms; i++) {
			for (int j = 0; j < number_of_atoms; j++) {
				double mass1 = prot -> getFromMassMatrix(i);
				double mass2 = prot -> getFromMassMatrix(j);
				for (int k = 0; k < 3; k++) {
					for (int l = 0; l < 3; l++) {
						hessianMatrix[i * 3 + k][j * 3 + l] = hessianMatrix[i * 3 + k][j * 3 + l] / (sqrt(mass1) * sqrt(mass2));
					}
				}
			}
		}
	}
	return hessianMatrix;
}

double EigenValAndVecCalculatorLapack::calculateDistance(double* xyz_atom1, double* xyz_atom2) {
	double dx = xyz_atom1[0] - xyz_atom2[0];
	double dy = xyz_atom1[1] - xyz_atom2[1];
	double dz = xyz_atom1[2] - xyz_atom2[2];

	return sqrt(pow(dx, 2) + pow(dy, 2) + pow(dz, 2));
}
