/*!
 * @file
 * @author  Lars Ackermann <matthias.ullmann@uni-bayreuth.de>
 * @version 1.0
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @section DESCRIPTION
 *
 * The EigenValAndVecCalculator class is an abstract class - DO NOT create an instance. It only ensures that all different variants of calculating
 * eigenvalues and eigenvectors provide the same results. Differences are only allowed in matters of performance.
 */

#include "Protein.h"
#include "SetupConstants.h"
#include "FileHandler.h"

#ifndef EIGENVALANDVECCALCULATOR_H_
#define EIGENVALANDVECCALCULATOR_H_

class EigenValAndVecCalculator {
public:

	/*!
	 * This is the constructor for EigenValAndVecCalculator which is abstract. So DO NOT create an instance.
	 * For using a calculator of this type, please check one of its implementations.
	 * e.g. \link<EigenValAndVecCalculatorLapack.h>EigenValAndVecCalculatorLapack\endlink
	 * @param prot1 This is the protein data structure which holds all relevant contents of the pqrm file.
	 * @param settings1 This parameter means the currently used settings. Most of them can be specified via the file "settings".
	 * @param fileHandler This class is responsible for input/output actions, e.g. reading the settings file.
	 * @return nothing to return
	 */
	EigenValAndVecCalculator(Protein* prot1, SetupConstants* settings1,
			FileHandler* fileHandler);

	/*!
	 * This is the standard destructor of an object.
	 * @return nothing to return
	 */
	virtual ~EigenValAndVecCalculator();

	//Set a function = 0 if you don't want to get an error if an instance of this class is used directly.
	//This is useful when you want to use a class as interface or in an abstract class for a not implemented method.
	//So there has to be another class which implements these "0"-methods.

	/*!
	 * This method calculates the kirchhoff matrix for a given model. It uses user specified interaction constants
	 * (e.g. kcovG) to decide whether or even how strong two atoms are connected. Each entry gives an answer to this
	 * question for two specific atoms.
	 * @param model The parameter specifies the model the kirchhoff matrix has to be calculated for.
	 * @return a kirchhoff matrix as two-dimensional array of type double.
	 */
	virtual double** buildKirchhoffMatrix(const char* model) = 0;

	/*!
	 * This method calculates the hessian matrix from a given kirchhoff matrix.
	 * @param kirchhoffMatrix This is the previously calculated kirchhoff matrix for the anisotropic model.
	 * @return the hessian matrix.
	 */
	virtual double** buildInteractionMatrixANM(double** kirchhoffMatrix) = 0;

	/*!
	 * This method calculates the distance between two given atoms. The coordinates used are specified in the pqrm file.
	 * @param xyz_atom1 This array contains the x-y-z coordinates for the first atom.
	 * @param xyz_atom2 This array contains the x-y-z coordinates for the second atom.
	 * @return the distance between the two atoms.
	 */
	virtual double calculateDistance(double* xyz_atom1, double* xyz_atom2) = 0;

	/*!
	 * This method calculates the eigenvalues and -vectors. Make sure that the methods
	 * EigenValAndVecCalculator::buildKirchhoffMatrix and buildKirchhoffMatrix::buildInteractionMatrixANM work properly.
	 */
	virtual void calculateAndPrintEigenValsAndVecs() = 0;

protected:
	Protein* prot;
	SetupConstants* settings;
	FileHandler* fileHandler;
};

#endif /* EIGENVALANDVECCALCULATOR_H_ */
